(ns de.phenomdevel.formalicious.core
  (:require
   [schema.core :as s]
   [re-frame.core :refer [subscribe]]
   [plumbing.core :refer-macros [letk] :refer [map-vals]]

   [de.phenomdevel.formalicious.handler]
   [de.phenomdevel.formalicious.subscription]

   [de.phenomdevel.formalicious.util :as u]
   [de.phenomdevel.formalicious.form :as form]
   [de.phenomdevel.formalicious.schema.shared :as shared]
   [de.phenomdevel.formalicious.schema.form :refer [FormSpec]]))


;; =============================================================================
;; Helper
;; TODO: Make sure buttons won't be subscribed ever
;; TODO: Make sure there are as few subscriptions as possible
(defn- fields->data
  "Takes form-spec and creates a data-map which represents the form-spec 1:1.
  Either with the provided data or default value \"\""
  [fields data-root data]
  (reduce
   (fn [acc field]
     (if (contains? acc (:id field))
       acc
       (letk [[id {data-path id}]
              field

              data
              (subscribe [:state/get-at (u/collify (conj data-root data-path))])

              value
              (if (nil? @data)
                ""
                @data)]

         (assoc acc id value))))
   data
   (filter #(not= :button (:type %)) fields)))



;; =============================================================================
;; Public API

(s/defn render :- shared/Hiccup
  ([form-spec :- FormSpec]
   (render form-spec {} {}))

  ([form-spec :- FormSpec
    data      :- s/Any]
   (render form-spec data {}))

  ([form-spec :- FormSpec
    data      :- s/Any
    opts      :- s/Any]
   (letk [[{data-root nil}]
          form-spec

          [{wrapper [:div]}]
          opts

          fields
          (->> form-spec
               :fields
               (mapv u/idify)
               (sort-by :pos)
               (vec))

          data*
          (fields->data fields data-root data)

          rendered-fields
          (->> fields
               (mapv (partial form/field (u/collify data-root) data*)))]

     (into wrapper rendered-fields))))


(s/defn render-fields :- {s/Keyword s/Any}
  "Takes a `form-spec` and `data`.
   It will produce a map just like the input `form-spec` but with
   the rendered html-elements as vals.

   You can use this function if you want to get the rendered html-elements in your view-code and
   handle it manually."
  [form-spec :- FormSpec
   data :- s/Any]
  (letk [[{data-root nil}]
         form-spec

         fields
         (->> form-spec
              :fields
              (reduce
               (fn [acc [k field]]
                 (assoc acc k (assoc field :id k)))
               {}))

         data*
         (fields->data (vals fields) data-root data)

         rendered-fields
         (map-vals (partial form/field (u/collify data-root) data*) fields)  ]

    rendered-fields))
