(ns de.phenomdevel.formalicious.form
  (:require
   [plumbing.core :refer-macros [letk]]
   [re-frame.core :refer [dispatch subscribe]]

   [de.phenomdevel.formalicious.util :as u]
   [de.phenomdevel.formalicious.helper :as h]))


;; =============================================================================
;; Common Helper

(defn- field-spec->attrs
  [field-spec data data-root data-path]
  (letk [[{on-change nil} {disabled nil} {on-click nil}]
         field-spec

         update-path
         (conj data-root data-path)

         default-attrs
         (reduce
          (fn [acc [k v]]
            (if v
              (assoc acc k v)
              acc))
          {}
          (select-keys field-spec [:id :type :multiple :class :checked :placeholder]))]

    (-> default-attrs
        (h/with-value data data-path)
        (h/with-disabled disabled)
        (h/with-on-click on-click)
        (h/with-on-change on-change update-path))))


;; =============================================================================
;; Field Specs

(defmulti field
  (fn [data-root data field-spec]
    (or (:type field-spec) :text)))


;; =============================================================================
;; Default

(defmethod field :default
  [_]
  [:div "Input-Type not implemented."])


;; =============================================================================
;; Text

(defmethod field :text
  [data-root data field-spec]
  (letk [[{id ""}
          {class nil}
          {data-path id}]
         field-spec

         attrs
         (field-spec->attrs field-spec data data-root data-path)

         input-field
         [:input attrs]]

    (-> {:class class}
        (h/with-input-field-wrapper)
        (h/with-label field-spec)
        (h/with-contents input-field))))


;; =============================================================================
;; Password

(defmethod field :password
  [data-root data field-spec]
  (letk [[{id ""}
          {class ""}
          {data-path id}]
         field-spec

         attrs
         (field-spec->attrs field-spec data data-root data-path)

         input-field
         [:input attrs]]

    (-> {:class class}
        (h/with-input-field-wrapper)
        (h/with-label field-spec)
        (h/with-contents input-field))))


;; =============================================================================
;; Textarea

(defmethod field :textarea
  [data-root data field-spec]
  (letk [[{id ""}
          {class nil}
          {data-path id}]
         field-spec

         attrs
         (field-spec->attrs field-spec data data-root data-path)

         input-field
         [:textarea attrs]]

    (-> {:class class}
        (h/with-input-field-wrapper)
        (h/with-label field-spec)
        (h/with-contents input-field))))


;; =============================================================================
;; Select

(defn field-spec->options
  [field-spec]
  (letk [[{options nil}
          {options-path nil}
          {options-transform nil}]
         field-spec
         _ (js/console.log (pr-str options))
         options
         (or options
             @(subscribe [:state/get-at (u/collify options-path)]))]

    (if-not options-transform
      options
      (mapv options-transform options))))

(defn ensure-id
  [option]
  (if-not (map? option)
    {:id option :label option}
    (letk [[{id nil} {label nil}]
           option]
      (if (nil? id)
        {:id label :label label}
        option))))

(defmethod field :select
  [data-root data field-spec]
  (letk [[{id ""}
          {class nil}
          {data-path id}
          {order-by :id}
          {multiple false}
          {options-label-key :label}]
         field-spec

         attrs
         (cond-> (field-spec->attrs field-spec data data-root data-path)
           multiple
           (update :value u/collify))

         options
         (->> field-spec
              (field-spec->options)
              (map ensure-id)
              (sort-by order-by))

         option-fields
         (for [option options]
           (letk [[{id ""}]
                  option]
             (-> {:value id}
                 (h/with-option-wrapper id)
                 (conj (str (get option options-label-key "NO VALUE"))))))

         select-field
         (-> attrs
             (h/with-select-field-wrapper)
             (h/with-contents (h/default-option field-spec) option-fields))]

    (-> {:class class}
        (h/with-input-field-wrapper)
        (h/with-contents select-field))))


;; =============================================================================
;; Button

(defmethod field :button
  [data-root data field-spec]
  (letk [[{id ""}
          {label ""}
          {class nil}]
         field-spec

         attrs
         (field-spec->attrs field-spec nil nil nil)]

    [:button
     attrs
     label]))


;; =============================================================================
;; Checkbox

(defmethod field :checkbox
  [data-root data field-spec]
  (letk [[{id ""}
          {class ""}
          {data-path id}]
         field-spec

         attrs
         (-> (field-spec->attrs field-spec data data-root data-path)
             (assoc :on-change
                    (fn [e]
                      (dispatch [:state/update-at (conj data-root data-path) (.-checked (.-target e))]))))

         checkbox-field
         [:input attrs]]

    (-> {:class class}
        (h/with-input-field-wrapper)
        (h/with-label field-spec)
        (h/with-contents checkbox-field))))


;; =============================================================================
;; Multiselect

(defmethod field :multiselect
  [data-root data field-spec]
  (letk [[{id ""}
          {data-path id}]
         field-spec

         update-path
         (conj data-root data-path)

         ;; TODO What if the user proviedes :select with :multiple true
         ;; The select-field won't recognize it and won't have the right on-change handler
         ;; Don't assoc on-change here. Make the on-change helper aware of the field-type
         field-spec*
         (-> field-spec
             (assoc :type :select
                    :multiple true
                    :on-change (fn [e]
                                 (let [options
                                       (->> e
                                            (.-target)
                                            (.-options)
                                            (array-seq)
                                            (remove #(not (.-selected %)))
                                            (mapv #(.-value %)))]

                                   (h/on-change-handler update-path options)))))]

    (field data-root data field-spec*)))

;; =============================================================================
;; Radiobutton

(defmethod field :radiobutton
  [data-root data field-spec]
  (letk [[{id ""}
          {class ""}
          {data-path id}]
         field-spec

         attrs
         (-> (field-spec->attrs field-spec data data-root data-path)
             (assoc :type :radio)
             (assoc :checked (= (name id) (get data data-path ""))))

         input-field
         [:input (assoc attrs :value id)]]

    (-> {:class class :key id}
        (h/with-input-field-wrapper)
        (h/with-label field-spec)
        (h/with-contents input-field))))


;; =============================================================================
;; Radiogroup

(defmethod field :radiogroup
  [data-root data field-spec]
  (letk [[{id ""}
          {class ""}
          ;; TODO configurable
          {options [{:id "1" :label "Eins"}
                    {:id "2" :label "Zwei"}]}
          {data-path id}]
         field-spec

         on-change
         (fn [e]
           (dispatch [:state/update-at (conj data-root data-path) (.-value (.-target e))]))

         attrs
         (field-spec->attrs field-spec data data-root data-path)

         radiogroup-field
         [:div.radiogroup
          (for [option options]
            (let [f-spec
                  (assoc option
                         :type :radiobutton
                         :data-path data-path)]

              (field data-root data f-spec)))]]

    (-> {:class class}
        (h/with-input-field-wrapper)
        (h/with-label field-spec)
        (h/with-contents radiogroup-field))))


;; =============================================================================
;; Datepicker

(defmethod field :date
  [data-root data field-spec]
  (letk [[{id ""}
          {class nil}
          {data-path id}]
         field-spec

         attrs
         (field-spec->attrs field-spec data data-root data-path)

         input-field
         [:input attrs]]

    (-> {:class class}
        (h/with-input-field-wrapper)
        (h/with-label field-spec)
        (h/with-contents input-field))))
