(ns de.phenomdevel.formalicious.core
  (:require
   [schema.core :as s]
   [re-frame.core :refer [subscribe]]
   [plumbing.core :refer-macros [letk]]

   [de.phenomdevel.formalicious.handler]
   [de.phenomdevel.formalicious.subscription]

   [de.phenomdevel.formalicious.util :as u]
   [de.phenomdevel.formalicious.form :as form]
   [de.phenomdevel.formalicious.schema.shared :as shared]
   [de.phenomdevel.formalicious.schema.form :refer [FormSpec Opts]]))


;; =============================================================================
;; Helper

(defn- fields->data
  [fields data-root data]
  (reduce
   (fn [acc field]
     (if (contains? acc (:id field))
       acc
       (letk [[id {data-path nil}]
              field

              data
              (when data-path
                (subscribe [:state/get-at (u/collify data-path)]))

              value
              (cond
                (or (nil? data) (nil? @data))
                ""

                :else
                @data)]

         (assoc acc id value))))
   data
   (filter #(not= :button (:type %)) fields)))


;; =============================================================================
;; Public API

(s/defn render :- shared/Hiccup
  "Takes a `form-spec` and `data` which will be rendered as react-components using re-frame.

Optionally you can provide `opts` which can contain additional options:
- `:wrapper` overrides the default hiccup-element wrapper [:div ...]."
  ([form-spec :- FormSpec
    data      :- {s/Any s/Any}]
   (render form-spec data {}))

  ([form-spec :- FormSpec
    data      :- {s/Any s/Any}
    opts      :- Opts]
   (letk [[{data-root nil}]
          form-spec

          [{wrapper [:div]}]
          opts

          fields
          (->> form-spec
               :fields
               (mapv u/idify)
               (sort-by :pos)
               (vec))

          data*
          (fields->data fields data-root data)

          rendered-fields
          (for [field fields]
            (let [field-data
                  (get data* (:id field))]

              [form/field (u/collify data-root) field-data field]))]

     (into wrapper rendered-fields))))


(s/defn render-fields :- {s/Keyword s/Any}
  "Takes a `form-spec` and `data`.
   It will produce a map just like the input `form-spec` but with
   the rendered html-elements as vals.

   You can use this function if you want to get the rendered html-elements in your view-code and
   handle it manually."
  [form-spec :- FormSpec
   data :- s/Any]
  (letk [[{data-root nil}]
         form-spec

         fields
         (->> form-spec
              :fields
              (reduce
               (fn [acc [k field]]
                 (assoc acc k (assoc field :id k)))
               {}))

         data*
         (fields->data (vals fields) data-root data)

         rendered-fields
         (for [field fields]
           (let [field-data
                 (get data* (:id field))]

             [form/field (u/collify data-root) field-data field]))]

    rendered-fields))
