(ns de.phenomdevel.formalicious.core
  (:require
   [schema.core :as s]
   [re-frame.core :refer [subscribe]]
   [plumbing.core :refer-macros [letk]]

   [de.phenomdevel.formalicious.handler]
   [de.phenomdevel.formalicious.subscription]

   [de.phenomdevel.formalicious.util :as u]
   [de.phenomdevel.formalicious.form :as form]
   [de.phenomdevel.formalicious.schema.shared :as shared]
   [de.phenomdevel.formalicious.schema.form :refer [FormSpec Opts]]))


;; =============================================================================
;; Helper

(defn- interpolate-data-root
  [data-root field]
  (letk [[{id ""}
          {data-path nil}]
         field

         value
         (if data-path
           (u/to-data-path data-path)
           (u/to-data-path data-root id))]

    (assoc field :data-path value)))

(defn- interpolate-value
  [data field]
  (if (= :button (:type field))
    field
    (letk [[{id ""}
            {data-path nil}]
           field

           value
           (if data-path
             @(subscribe [:state/get-at (u/collify data-path)])
             (get data id ""))]

      (assoc field :value value))))

(defn- transform-form-spec
  [form-spec data]
  (letk [[{fields {}}
          {data-root []}]
         form-spec]

    (->> fields
         (mapv u/idify)
         (sort-by :pos)
         (vec)
         (mapv (partial interpolate-value data))
         (mapv (partial interpolate-data-root data-root)))))


;; =============================================================================
;; Public API

(s/defn render :- shared/Hiccup
  "Takes a `form-spec` and `data` which will be rendered as react-components using re-frame.

Optionally you can provide `opts` which can contain additional options:
- `:wrapper` overrides the default hiccup-element wrapper [:div ...]."
  ([form-spec :- FormSpec
    data      :- {s/Any s/Any}]
   (render form-spec data {}))

  ([form-spec :- FormSpec
    data      :- {s/Any s/Any}
    opts      :- Opts]
   (letk [[{wrapper [:div]}]
          opts

          fields
          (transform-form-spec form-spec data)

          rendered-fields
          (for [field fields]
            [form/field field])]

     (into wrapper rendered-fields))))

;; TODO: Make it work again
#_(s/defn render-fields :- {s/Keyword s/Any}
    "Takes a `form-spec` and `data`.
   It will produce a map just like the input `form-spec` but with
   the rendered html-elements as vals.

   You can use this function if you want to get the rendered html-elements in your view-code and
   handle it manually."
    [form-spec :- FormSpec
     data :- s/Any]
    (letk [[{data-root nil}]
           form-spec

           fields
           (transform-form-spec form-spec data)

           rendered-fields
           (reduce
            (fn [acc field]
              (let [field-spec
                    (->> field
                         (u/idify)
                         (interpolate-value data)
                         (interpolate-data-root data-root))

                    rendered-field
                    [form/field field-spec (:value field-spec)]]

                (assoc acc (:id field-spec) rendered-field)))
            {}
            fields)]

      (assoc form-spec :fields rendered-fields)))
