(ns example.views.select
  (:require
   [re-frame.core :refer [subscribe]]

   [example.util :as u]

   [de.phenomdevel.formalicious.core :as formalicious]
   [clojure.string :as str]))

;; =============================================================================
;; Form-Spec

(def base-form-spec
  {:data-root
   [:example :form-data :select]

   :fields
   {}})


;; =============================================================================
;; Sub Views

(defn- normal-select
  [form-data]
  (let [form-spec
        (-> base-form-spec
            (assoc-in [:fields :normal-select]
                      {:type :select
                       :label "Normal Select"
                       :options ["1" "2" "3"]
                       :order-by :id}))]

    [:div.normal-select
     [:h3 "Normal Select"]
     "Here is the `form-spec` used to render the `form-field`:"
     (u/data->html form-spec)
     [formalicious/render form-spec form-data]
     [:br]
     "Why can we order by `:id` if there is no `id` within any option?"
     [:br]
     "Formalicious will produce maps for each option if there are not already maps supplied."
     [:br]
     "If you supply `strings` they will look like:"
     (u/data->html {:id "option-string" :label "option-string"})
     "Therefore you could order by id or label by default which will have the same behaviour."]))

(defn- normal-multiselect
  [form-data]
  (let [form-spec
        (-> base-form-spec
            (assoc-in [:fields :normal-multiselect]
                      {:type :multiselect
                       :multiple true
                       :label "Normal Multiselect"
                       :options ["1" "2" "3"]}))]

    [:div.normal-multiselect
     [:h3 "Normal Multiselect"]
     "Here is the `form-spec` used to render the `form-field`:"
     (u/data->html form-spec)
     [formalicious/render form-spec form-data]]))

(defn- subscribed-select
  [form-data]
  (let [form-spec
        (-> base-form-spec
            (assoc-in [:fields :subscribed-select]
                      {:type :select
                       :label "Subscribed Select"
                       :options (:normal-multiselect form-data)}))]

    [:div.subscribed-select
     [:h3 "Subscribed Select"]
     "Here is the `form-spec` used to render the `form-field`:"
     (u/data->html form-spec)
     [formalicious/render form-spec form-data]
     [:br]
     "This select field is a special case. Try change the multiselect value above and you'll notice a change in possible select options of this element."]))

(defn- select-with-transformation
  [form-data]
  (let [form-spec
        (-> base-form-spec
            (assoc-in [:fields :select-with-transformation]
                      {:type :select
                       :label "Select With Transformation"
                       :options [{:id "1" :name "Option 1" :value "opt1"}
                                 {:id "2" :name "Option 2" :value "opt2"}
                                 {:id "3" :name "Option 3" :value "opt3"}]
                       :option-label-key :name
                       :order-by :id
                       :option-value-key :value
                       :option-transform #(update % :name str/reverse)}))]

    [:div.select-with-transformation
     [:h3 "Select With Transformation"]
     "Here is the `form-spec` used to render the `form-field`:"
     (u/data->html form-spec)
     [formalicious/render form-spec form-data]]))


;; =============================================================================
;; View

(defn view
  []
  (let [!form-data
        @(subscribe [:form/data :select])]

    [:div
     [:h2
      "Select Examples"
      [:br]
      "Here is the `app-state` for this example:"
      (u/data->html !form-data)]
     [:hr]
     [normal-select !form-data]
     [:hr]
     [normal-multiselect !form-data]
     [:hr]
     [subscribed-select !form-data]
     [:hr]
     [select-with-transformation !form-data]]))
