(ns de.sveri.friendui-datomic.db
  (:require [datomic.api :as d]
            [de.sveri.friendui-datomic.globals :as glob]
            [de.sveri.friendui.service.user :as friend-service]
            [de.sveri.clojure.commons.datomic.db :as comm-db]))

(def ^:dynamic *FrienduiDatomicImpl* nil)

(defprotocol FrienduiDatomic
  "Defines methods to acces datomic storage."
  (find-by-attr-and-search-string [this db-val attr string])
  (update-entity-by-id [this db-conn id data-map])
  (create-entity [this data-map])
  (insert-entity [this db-conn entity])
  (get-by-id [this db-val id]
             "Should return a map containing the username and role of this user like this:
             {:username username :roles #{role}}")
  (find-all-from-column [this db-val column-query]
                        "Updates the user with the given data map of the form: {:user/activated boolean :user/role :user/free}")
  (find-by-username-query [this db-val username]))

;predefined queries
(def pre-find-by-username-query `[:find ?e :where [?e ~glob/username-kw]])

(defn get-usermap-by-username [db-val username]
  (find-by-username-query *FrienduiDatomicImpl* db-val username))

(defn get-loggedin-user-map [db-val]
  (get-usermap-by-username db-val (friend-service/get-logged-in-username)))

(defn login-user [db-val username]
  (when-let [user-map (get-usermap-by-username (db-val) username)]
    (when (friend-service/is-user-activated? user-map)
      {:username username :roles #{(glob/role-kw user-map)} :password (glob/pw-kw user-map)})))

(defn activate-account [db-conn db-val activationid]
  (when-let [user-id (ffirst (find-by-attr-and-search-string *FrienduiDatomicImpl* db-val glob/activationid-kw activationid))]
    (update-entity-by-id *FrienduiDatomicImpl* db-conn user-id {glob/activated-kw true})))

(defn create-new-user [db-conn email password role activationid]
  {:pre [(some? activationid)]}
  (let [user-map (create-entity *FrienduiDatomicImpl* {glob/username-kw email glob/pw-kw password glob/role-kw role glob/activationid-kw activationid})]
    (insert-entity *FrienduiDatomicImpl* db-conn user-map)))

(defn get-all-users
  "Returns a list of user maps with all data available in database, without the password."
  [db-val]
  (sort-by glob/username-kw
           ;(mapv #(dissoc (into {} (get-by-id *FrienduiDatomicImpl* db-val (first %))) glob/pw-kw)
           (mapv #(dissoc (into {} (get-by-id *FrienduiDatomicImpl* db-val (first %))) glob/pw-kw)
                 (find-all-from-column *FrienduiDatomicImpl* db-val pre-find-by-username-query))))

(defn get-user-for-activation-id [db-val activationid]
  (when-let [user-id (ffirst (find-by-attr-and-search-string *FrienduiDatomicImpl* db-val glob/activationid-kw activationid))]
    (let [user (get-by-id *FrienduiDatomicImpl* db-val user-id)]
      {:username (glob/username-kw user)
       :roles    #{(glob/role-kw user)}})))

(defn update-user [db-conn username data-map]
  @(d/transact db-conn [(merge {:db/id [glob/username-kw username]} data-map)]))

(defn username-exists? [db-val username]
  (if (ffirst (find-by-attr-and-search-string *FrienduiDatomicImpl* db-val glob/username-kw username)) true false))


(defrecord FrienduiDatomicImpl []
  FrienduiDatomic
  (find-by-attr-and-search-string [this db-val attr string]
    (comm-db/find-by-attr-and-search-string db-val attr string))
  (update-entity-by-id [this db-conn id data-map]
    (comm-db/update-entity-by-id db-conn id data-map))
  (create-entity [this data-map]
    (comm-db/create-entity glob/partition-id data-map))
  (insert-entity [this db-conn entity]
    (comm-db/insert-entity db-conn entity))
  (get-by-id [this db-val id]
    (comm-db/get-by-id-lazy-ref db-val id))
  (find-all-from-column [this db-val column-query]
    (comm-db/find-all-from-column db-val column-query))
  (find-by-username-query [this db-val username]
    (when-let [entity (comm-db/find-entity-by-attr-and-search-string db-val glob/username-kw username)]
      (comm-db/get-map-from-entity entity))))


