(ns de.sveri.friendui-datomic.db
  (:require [datomic.api :as d]
            [de.sveri.friendui-datomic.globals :as glob]
            [de.sveri.friendui.service.user :as friend-service]
            [de.sveri.clojure.commons.datomic.db :as comm-db]))

;predefined queries
(def pre-find-by-username-query `[:find ?e :where [?e ~glob/username-kw]])

(defn get-usermap-by-username [db-val username & [pw]]
  (when-let [entity (comm-db/find-entity-by-attr-and-search-string db-val glob/username-kw username)]
    (let [user-map (into {} (comm-db/get-map-from-entity entity))]
      (if pw user-map (dissoc user-map glob/pw-kw)))))

(defn get-loggedin-user-map [db-val]
  (get-usermap-by-username db-val (friend-service/get-logged-in-username)))

(defn get-loggedin-user-role [db-val]
  (glob/role-kw (get-loggedin-user-map db-val)))

(defn login-user [db-val username]
  (when-let [user-map (get-usermap-by-username (db-val) username true)]
    (when (friend-service/is-user-activated? user-map)
      {:username username :roles #{(glob/role-kw user-map)} :password (glob/pw-kw user-map)})))

(defn activate-account [db-conn db-val activationid]
  (when-let [user-id (ffirst (comm-db/find-by-attr-and-search-string db-val glob/activationid-kw activationid))]
    (comm-db/update-entity-by-id db-conn user-id {glob/activated-kw true})))

(defn create-new-user [db-conn email password role activationid]
  {:pre [(some? activationid)]}
  (let [user-map (comm-db/create-entity glob/partition-id
                                        {glob/username-kw     email
                                         glob/pw-kw           password
                                         glob/role-kw         role
                                         glob/activationid-kw activationid})]
    (comm-db/insert-entity db-conn user-map)))

(defn get-all-users
  "Returns a list of user maps with all data available in database, without the password."
  [db-val]
  (sort-by glob/username-kw
           (mapv #(dissoc (into {} (comm-db/get-by-id-lazy-ref db-val (first %))) glob/pw-kw)
                 (comm-db/find-all-from-column db-val pre-find-by-username-query))))

(defn get-user-for-activation-id [db-val activationid]
  (when-let [user-id (ffirst (comm-db/find-by-attr-and-search-string db-val glob/activationid-kw activationid))]
    (let [user (comm-db/get-by-id-lazy-ref db-val user-id)]
      {:username (glob/username-kw user)
       :roles    #{(glob/role-kw user)}})))

(defn update-user [db-conn username data-map]
  @(d/transact db-conn [(merge {:db/id [glob/username-kw username]} data-map)]))

(defn username-exists? [db-val username]
  (if (ffirst (comm-db/find-by-attr-and-search-string db-val glob/username-kw username)) true false))

(defn get-old-pw-hash [db-val]
  (glob/pw-kw (get-usermap-by-username db-val (friend-service/get-logged-in-username) true)))

(defn change-password [db-conn new-pw]
  (update-user db-conn (friend-service/get-logged-in-username) {glob/pw-kw new-pw}))