(ns sock.client
  (:require [cljs.core.async :refer [chan dropping-buffer put! >! <! close! map< filter<]]
            [cljs.edn :refer [read-string]]
            [goog.events :as events])
  (:require-macros [cljs.core.async.macros :refer [go go-loop]]))


(defn ws
  "Connects to the given adress via a webSocket and
  returns a set of channels to communicate with the socket.

  This collection is of the form
  `{:in dropping-chan :out dropping-chan :log chan}`.

  Closing the `:out` channel closes the websocket."
  [address & {:keys [in out log]
              :or {in  (chan (dropping-buffer 1024))
                   out (chan (dropping-buffer 1024))
                   log (chan (dropping-buffer 64))}}]
  (let [socket (js/WebSocket. address ["text" "edn"])]
    (doto socket
      (aset "on-open" (fn [event]
                           (let [msg (.-data event)]
                             (put! log {:event :open}))))
      (aset "on-message" (fn [event]
                           (let [msg (.-data event)]
                             (put! out (read-string msg)))))
      (aset "on-error"   (fn [event]
                           (let [error (.-data event)]
                             (put! log {:event :error :status error}))))
      (aset "on-close"   (fn []
                           (put! log {:event :close
                                      :origin (if (clojure.core.async.impl.protocols/closed? out) :local :remote)})
                           (close! in)
                           (close! out)
                           (close! log))))
    (go-loop []
             (if-let [msg (<! out)]
               (do (.send socket (pr-str msg))
                 (recur))
               (do (.close socket))))
    chans))
