(ns sock.client
  (:require [cljs.core.async :refer [chan dropping-buffer put! >! <! close! map< filter<]]
             cljs.core.async.impl.protocols
            [cljs.reader :as reader]
            [goog.events :as events])
  (:require-macros [cljs.core.async.macros :refer [go go-loop]]))


(defn ws
  "Connects to the given adress via a webSocket and
  returns a set of channels to communicate with the socket.

  This collection is of the form
  `{:in dropping-chan :out dropping-chan :log dropping-chan}`.

  Closing the `:out` channel closes the websocket."
  [address & {:keys [in out log]
              :or {in  (chan (dropping-buffer 1024))
                   out (chan (dropping-buffer 1024))
                   log (chan (dropping-buffer 1024))}}]
  (let [socket (js/WebSocket. address)]
    (doto socket
      (aset "onopen" (fn [event]
                           (let [msg (.-data event)]
                             (put! log {:event :open}))))
      (aset "onmessage" (fn [event]
                           (let [msg (.-data event)]
                             (put! in (reader/read-string msg)))))
      (aset "onerror"   (fn [event]
                           (let [error (.-data event)]
                             (put! log {:event :error :status error}))))
      (aset "onclose"   (fn []
                           (put! log
                                 {:event :close
                                  :origin (if (cljs.core.async.impl.protocols/closed? out)
                                            :local
                                            :remote)})
                           (close! in)
                           (close! out)
                           (close! log))))
    (go-loop []
             (if-let [msg (<! out)]
               (do (.send socket (pr-str msg))
                 (recur))
               (do (.close socket))))
    {:in in
     :out out
     :log log}))
