(ns de.uni-koblenz.ist.funtg.funtl
  (:use de.uni-koblenz.ist.funtg.core)
  (:require clojure.set)
  (:require clojure.pprint)
  (:import
   (de.uni_koblenz.jgralab Graph Vertex Edge M1ClassManager)
   (de.uni_koblenz.jgralab.codegenerator CodeGeneratorConfiguration)
   (de.uni_koblenz.jgralab.schema AggregationKind Attribute
                                  AttributedElementClass GraphClass EdgeClass
                                  Schema VertexClass)
   (de.uni_koblenz.jgralab.schema.impl SchemaImpl)))

;;* TODO List

;; - The usage of those dynamic Vars is not really optimal.  There's no good
;;   reason why you shouldn't be able to use create-vertices! outside of a
;;   transformation.  The mapping should be passed along...
;;
;; - Allow for many target graphs!
;;
;; - Implement Collection and Map attribute creation: List<String>,
;;   Map<Integer,String>, conversion of attribute values is already done.
;;
;; - Implement Enum domains (create-enum-domain 'Foo :BAR :BAZ :QUUX), similar
;;   to record.  Use fn (enum elem 'Foo :BAZ) that returns the right literal.
;;
;; - Implement Record domains (create-record-domain 'Foo {:c1 String, :c2
;;   Integer}), attribute value conversion already done (record 'Foo {:c1 "Hi",
;;   :c2 17})

;;* FunTL

;;** Dynamic vars

(def ^:dynamic
  $target-graph  nil)
(def ^:dynamic
  $target-schema nil)
(def ^:dynamic
  $img           nil)
(def ^:dynamic
  $arch          nil)
(def ^:dynamic
  $on-graph-fns  nil)


;;** Utility functions

(defn- aec-internal
  "Gets the AttributedElementClass by the given qname.
  Can only be called inside a deftransformation."
  [qname]
  (let [aec (.getAttributedElementClass
             ^Schema $target-schema (name qname))]
    (or aec
        (throw (RuntimeException.
                ^String (format "No such AttributedElementClass %s."
                                qname))))))

(defn- attr-internal
  "Returns the Attribute given by its qualified name.
  Can only be called inside a deftransformation."
  [qname]
  (let [[aec-qname attr-name _] (split-qname (name qname))
        ^AttributedElementClass aec (aec-internal aec-qname)]
    (or (.getAttribute aec attr-name)
        (throw (RuntimeException.
                ^String (format "No such Attribute %s at AttributedElementClass %s."
                                (name qname)
                                aec))))))

(defn- dom-internal
  "Returns the Domain given by its qualified name.
  Can only be called inside a deftransformation."
  [qname]
  (.getDomain ^Schema $target-schema (name qname)))

(defn- merge-into-mappings
  "Merges into oldmap the new mappings for cls (an GraphElementClass)."
  [oldmap ^AttributedElementClass cls new]
  (let [kvs (flatten
             (map (fn [c]
                    (let [old (oldmap c)
                          isect (seq (clojure.set/intersection
                                      (into #{} (keys old))
                                      (into #{} (keys new))))]
                      (when isect
                        (throw (RuntimeException.
                                ^String (format "The keys %s already exist in img/arch"
                                                isect))))
                      [c (merge old new)]))
                  (conj (filter #(not (.isInternal ^AttributedElementClass %))
                                (.getAllSuperClasses cls))
                        cls)))]
    (apply assoc oldmap kvs)))

(defn- img-internal
  "Returns the image of arch for AttributedElementClass aec.
  Can only be called inside a deftransformation."
  [aec arch]
  ((@$img aec) arch))

(defn img
  "Returns the map from archetypes to images for aec.
  aec can be given as string, symbol, or keyword denoting its qualified name."
  [aec]
  (@$img (aec-internal aec)))

(defn- arch-internal
  "Returns the archetype of img for AttributedElementClass aec.
  Can only be called inside a deftransformation."
  [aec img]
  ((@$arch aec) img))

(defn arch
  "Returns the map from images to archetypes for aec.
  aec can be given as string, symbol, or keyword denoting its qualified name."
  [aec]
  (@$arch (aec-internal aec)))

;;** Instance only functions

(defn create-vertices!
  "Creates one vertex of type cls for each archetype in archs.
  archs has to be a collection or function resulting in a collection.
  Returns a map from archetypes to images.
  Can only be called inside a deftransformation."
  [cls archs]
  {:pre [cls (fn? archs)]}
  (swap! $on-graph-fns conj
         (fn []
           (let [^VertexClass vc (aec-internal cls)]
             (loop [a (archs)
                    m (transient {})]
               (if (seq a)
                 (recur (rest a)
                        (assoc! m (first a) (create-vertex! $target-graph cls)))
                 (let [img (persistent! m)
                       ;; TODO: This is very expensive!  Better construct arch directly!
                       arch (clojure.set/map-invert img)]
                   (assert (== (count img) (count arch)) "img/arch counts don't match")
                   (swap! $img  merge-into-mappings vc img)
                   (swap! $arch merge-into-mappings vc arch)
                   img)))))))

(defn create-edges!
  "Creates one edge of type cls for each archetype in archs.
  archs has to be a collection of triples or a function resulting therein.
  Each triple has the form [arch start end] where arch is the archetype of the
  new edge, start is the new edge's start vertex archetype, and end is the new
  edge's end vertex archetype.
  Returns a map from archetypes to images.
  Can only be called inside a deftransformation."
  [cls archs]
  {:pre [cls (fn? archs)]}
  (swap! $on-graph-fns conj
         (fn []
           (let [^EdgeClass ec (aec-internal cls)
                 saec (-> ec (.getFrom) (.getVertexClass))
                 eaec (-> ec (.getTo)   (.getVertexClass))]
             (loop [a (archs), m (transient {})]
               (let [triple (first a)]
                 (if (seq a)
                   (recur (rest a)
                          (assoc! m
                                  (first triple)
                                  (create-edge! cls
                                                (img-internal saec (second triple))
                                                (img-internal eaec (nth triple 2)))))
                   (let [img (persistent! m)
                         ;; TODO: This is very expensive!  Better construct arch directly!
                         arch (clojure.set/map-invert img)]
                     (assert (== (count img) (count arch)) "img/arch counts don't match")
                     (swap! $img  merge-into-mappings ec img)
                     (swap! $arch merge-into-mappings ec arch)
                     img))))))))

(defn set-values!
  "Set the attr of valmap's keys images to the corresponding value.
  attr is a qualified name (foo.Bar.baz) and valmap a map or a function resulting
  in a map.
  Can only be called inside a deftransformation."
  [a valmap]
  {:pre [a (fn? valmap)]}
  (swap! $on-graph-fns conj
         (fn []
           (let [^Attribute a (attr-internal a)
                 aec (.getAttributedElementClass a)
                 name (.getName a)]
             (doseq [entry (valmap)]
               (set-value! (img-internal aec (first entry)) name (second entry)))))))

;;** Schema & instance functions

;;*** Creating VertexClasses

(defn- create-vc!
  [{:keys [qname abstract]}]
  (-> (.getGraphClass ^Schema $target-schema)
      (doto (.createVertexClass (name qname))
        (.setAbstract (boolean abstract)))))

(defn create-vertex-class!
  "Creates VertexClass + instances.
  The map given as first argument provides the schema properties.
  For archs, see function create-vertices!."
  [{:keys [qname abstract]
    :or {abstract false}
    :as props}
   & [archs]]
  {:pre [qname
         (if abstract (nil? archs) (fn? archs))]}
  (create-vc! {:qname qname :abstract abstract})
  (when archs
    (create-vertices! qname archs)))

;;*** Creating EdgeClasses

(defn- create-ec!
  [{:keys [qname abstract
           from from-multis from-role from-kind
           to to-multis to-role to-kind]}]
  (-> (.getGraphClass ^Schema $target-schema)
      (doto (.createEdgeClass (name qname)
                              (aec-internal from) (first from-multis)
                              (second from-multis) from-role from-kind
                              (aec-internal to) (first to-multis)
                              (second to-multis) to-role to-kind)
        (.setAbstract (boolean abstract)))))

(defn create-edge-class!
  "Creates an EdgeClass + instances.
  The map given as first argument provides the schema properties.
  For archs, see function create-edges!."
  [{:keys [qname abstract
           from from-multis from-role from-kind
           to   to-multis   to-role   to-kind]
    :or {abstract false
         from-multis [0, Integer/MAX_VALUE]
         from-role ""
         from-kind AggregationKind/NONE
         to-multis [0, Integer/MAX_VALUE]
         to-role ""
         to-kind AggregationKind/NONE}
    :as props}
   & [archs]]
  {:pre [qname to from
         (if abstract (nil? archs) (fn? archs))]}
  (create-ec! {:qname qname :abstract abstract
               :from from :from-multis from-multis :from-role (name from-role) :from-kind from-kind
               :to   to   :to-multis   to-multis   :to-role   (name to-role)   :to-kind   to-kind})
  (when archs
    (create-edges! qname archs)))

;;*** Creating Attributes

(defn- create-attr!
  [{:keys [qname domain default]}]
  (let [[qn a _] (split-qname qname)
        elem     ^AttributedElementClass (aec-internal qn)]
    (.addAttribute elem a (dom-internal domain))))

(defn create-attribute!
  "Creates an attribute and sets values.
  The map given as first argument determines the schema properties.
  For valmap, see set-values!."
  [{:keys [qname domain default] :as props}
   & [valmap]]
  {:pre [qname domain]}
  (create-attr! props)
  (when valmap
    (set-values! qname valmap)))

;;*** Creating type hierarchies

(defn add-sub-class!
  "Makes all subs sub-classes of super."
  [super & subs]
  (let [s (aec-internal super)]
    (if (isa? (class s) VertexClass)
      (doseq [sub subs]
        (.addSuperClass ^VertexClass (aec-internal sub) ^VertexClass s))
      (doseq [sub subs]
        (.addSuperClass ^EdgeClass (aec-internal sub) ^EdgeClass s)))))

(defn add-super-class!
  "Makes all supers super-classes of sub."
  [sub & supers]
  (let [s (aec-internal sub)]
    (if (isa? (class s) VertexClass)
      (doseq [super supers]
        (.addSuperClass ^VertexClass s ^VertexClass (aec-internal super)))
      (doseq [super supers]
        (.addSuperClass ^EdgeClass s ^EdgeClass (aec-internal super))))))

;;** The transformation macro itself

;; TODO: Generalize to allow for multiple target graphs.  Hm, thinking 'bout
;; it, maybe I should drop all that magic with $target-graph/schema and stuff
;; and make those parameters.  But then I have to think about how the target
;; graph can be bound before it exists (maybe delay/force...).
(defmacro deftransformation
  "Create a new transformation named name with docstring doc, the given
  params (input graph args), and the given body."
  ;; Nicer arglist in doc
  {:arglists '([name doc [params*] & body])}
  [name doc in-graphs & body]
  `(defn ~name
     ~(str doc
           "\n\n  The last parameter out defines the target graph.

    - If a graph is given, use that graph as target graph.
    - If a schema is given, use a new graph of this schema as target graph.
    - If a vector [<sname> <gcname>] is given, create a new schema (and graph)
      with the given schema and graph class name.

  The generated transformation function returns the target graph.")
     ;; Nicer arglist for documentation
     {:arglists
      '([~@(vec in-graphs) ~'out])}
     [~@(vec in-graphs) out#]
     (binding [$arch          (atom {})
               $img           (atom {})
               $on-graph-fns  (atom [])
               $target-schema
               (cond
                (instance? Schema out#) out#
                (instance? Graph  out#) (.getSchema ^Graph out#)
                (vector? out#) (let [sqname# (first out#)
                                     [prefix# sname#] (split-qname sqname#)]
                                 (or
                                  (try
                                    (-> (de.uni_koblenz.jgralab.M1ClassManager/instance sqname#)
                                        (.loadClass sqname#)
                                        (.getMethod "instance" (into-array Class []))
                                        (.invoke nil (into-array Object [])))
                                    (catch Exception e# nil))
                                  (doto (SchemaImpl. sname# prefix#)
                                    (.createGraphClass (name (second out#))))))
                :default (throw (RuntimeException.
                                 (str "No target graph, target schema, "
                                      " or vector [sname, gcname] given"))))]
       (let [existing# (> 0 (.getVertexClassCount (.getGraphClass ^Schema $target-schema)))]
         (if-not existing#
           (do ~@body)
           (println "Using existing schema" (.getQualifiedName ^Schema $target-schema)))
         (binding [$target-graph (if (instance? Graph out#)
                                   out#
                                   (do
                                     (when-not existing#
                                       (.compile ^Schema $target-schema
                                                 CodeGeneratorConfiguration/MINIMAL))
                                     (create-graph $target-schema
                                                   (str "TransformationCreated-"
                                                        (System/currentTimeMillis)))))]
           (doseq [f# @$on-graph-fns] (f#))
           $target-graph)))))

