(ns de.uni-koblenz.ist.funtg.core
  (:use ordered.set)
  (:require clojure.string)
  (:import
   (java.awt.event KeyEvent KeyListener)
   (java.util Collection)
   (java.lang.reflect Method)
   (javax.swing JFrame JScrollPane JLabel ImageIcon JOptionPane WindowConstants)
   (de.uni_koblenz.jgralab AttributedElement Graph GraphElement Vertex Edge
			   EdgeDirection GraphIO Record M1ClassManager
                           ImplementationType)
   (de.uni_koblenz.jgralab.schema AggregationKind Schema Domain RecordDomain
                                  AttributedElementClass NamedElement
                                  GraphClass VertexClass EdgeClass Attribute
                                  GraphElementClass)
   (de.uni_koblenz.jgralab.utilities.tg2dot Tg2Dot)
   (de.uni_koblenz.jgralab.utilities.tg2dot.dot GraphVizProgram GraphVizOutputFormat)
   (de.uni_koblenz.jgralab.codegenerator CodeGeneratorConfiguration)
   (de.uni_koblenz.jgralab.impl ConsoleProgressFunction)
   (org.pcollections ArrayPMap ArrayPSet ArrayPVector)))

;;* Utility Functions and Macros

;;** General purpose

(defn member?
  "Returns true, if e is a member of coll."
  [e coll]
  (some #(= e %) coll))

(defn xor
  "Logical XOR function."
  ([] false)
  ([f & r]
     (loop [t false, f (conj r f)]
       (if(seq f)
         (let [fv (first f)]
           (cond
            (and t fv)       false
            (and (not t) fv) (recur true (rest f))
            :else            (recur t (rest f))))
         t))))

(defn xor-fn
  "Takes a set of predicates and returns a function f that returns logical
  true if exactly one of the predicates returns true."
  [& ps]
  (fn xor-fn-1 [& args]
    (reduce xor (map #(apply % args) ps))))

;;** Debugging

;; Deleted the home-brewn JoC-based debugging stuff.  The same (and more) is
;; already available using swank.core/break and swank.core/local-bindings.


;;** Graph utilities

(defn show-graph
  "Show graph g in a JFrame, possibly with reversed edges (default false)."
  ([^Graph g]
     (show-graph g false))
  ([^Graph g reversed]
     (if (> (+ (.getVCount g) (.getECount g)) 600)
       (do (println "Graph too big for visualizing!") nil)
       (let [img (-> (doto ^Tg2Dot (Tg2Dot.)
                           (.setGraph ^Graph g)
                           (.setReversedEdges reversed))
                     (.convertToGraphVizImageIcon
                      (-> (GraphVizProgram.)
                          (.outputFormat GraphVizOutputFormat/PNG))))
             label (JLabel. img)
             frame (JFrame. (str "Graph: " (.getId g)))
             scale (atom 1.0)]
         (doto frame
           (-> (.getContentPane)
               (.add (JScrollPane. label)))
           (.setDefaultCloseOperation WindowConstants/DISPOSE_ON_CLOSE)
           (.addKeyListener
            (proxy [KeyListener] []
              (keyPressed [^KeyEvent e]
                (try
                  (let [kc (.getKeyCode e)
                        sf (cond
                            (== kc KeyEvent/VK_PLUS)   0.1
                            (== kc KeyEvent/VK_MINUS) -0.1
                            :default 0)
                        i  (.getImage img)
                        io (.getImageObserver img)]
                    (when-not (== sf 0)
                      (swap! scale + sf)
                      (.setIcon label (ImageIcon. (.getScaledInstance
                                                   i
                                                   (int (* @scale (.getWidth i io)))
                                                   (int (* @scale (.getHeight i io)))
                                                   (int java.awt.Image/SCALE_SMOOTH))))
                      (.repaint frame)))
                  (catch Throwable ex
                    (JOptionPane/showMessageDialog frame (.getMessage ex)))))
              (keyReleased [e])
              (keyTyped [e])))
           (.show)
           (.pack))))))

(defn tgtree
  [g]
  (-> (de.uni_koblenz.jgralab.utilities.tgtree.TGTree. g)
      (.setVisible true)))

(defn dot-graph
  [^Graph g reversed file & reversed-ecs]
  (de.uni_koblenz.jgralab.utilities.tg2dot.Tg2Dot/convertGraph
   g file reversed
   de.uni_koblenz.jgralab.utilities.tg2dot.dot.GraphVizOutputFormat/PDF
   (into-array Class reversed-ecs)))

(defn load-graph
  "Loads a graph from file.
If the schema is not already in memory or in the CLASSPATH, then code will be
generated using cg-config (defaulting to CodeGeneratorConfiguration/MINIMAL),
and it will be compiled in memory."
  ([file]
     (load-graph file CodeGeneratorConfiguration/MINIMAL))
  ([file cg-config]
     (GraphIO/loadSchemaAndGraphFromFile
      file cg-config (ConsoleProgressFunction. "Loading"))))

(defn load-schema
  "Loads a schema from file and compiles it if neccessary using the given config.
  cg-config defaults to CodeGeneratorConfiguration/MINIMAL."
  ([file]
     (load-schema file CodeGeneratorConfiguration/MINIMAL))
  ([file ^CodeGeneratorConfiguration cg-config]
     (let [^Schema s (GraphIO/loadSchemaFromFile file)]
       (try
         (.getM1Class (.getGraphClass s))
         (catch Exception _
           (println "Compiling schema" (.getQualifiedName s))
           (.compile s cg-config)))
       s)))

(defn save-graph
  "Saves g to file."
  [^Graph g ^String file]
  (GraphIO/saveGraphToFile g file (ConsoleProgressFunction. "Saving")))

(defn qname?
  "Returns true, if the given thing n is possibly a schema element name.
Only checks, if the given thing is a symbol, a keyword, or a string."
  [n]
  (or (symbol? n) (keyword? n) (string? n)))

(defn type-spec?
  "Returns true, if the given thing n is possibly a type specification.
     pkg.Foo
     :Bar!
     \"bla.Bla\"
     [Foo Bar !Baz]
     [:and !Foo !Bar]"
  [n]
  (or (qname? n)
      (and (vector? n)
           (let [x (first n)]
             (or (qname? x) (= x :and) (= x :or))))))

(defn split-qname
  "Given a qualified name qn as string, symbol, or keyword, returns a vector of
  the form [\"foo.baz\" \"Bar\" \"foo.baz.Bar\"], i.e., package, simple name,
  qname.

  For an attribute foo.baz.Bar.attr it's [\"foo.baz.Bar\" \"attr\" \"foo.baz.Bar.attr\"]."
  [qn]
  (let [qstr (name qn)
        liod (.lastIndexOf qstr ".")]
    (if (== liod -1)
      ["" qstr qstr]
      (let [sn (subs qstr (inc liod))
            pn (subs qstr 0 liod)]
        [pn sn qstr]))))

(defn- get-modifiers
  [^String name]
  (let [neg (.startsWith name "!")
        ext (.endsWith   name "!")
        name (.substring name
                         (if neg 1 0)
                         (if ext (dec (.length name)) (.length name) ))]
    [neg name ext]))

(defmacro dograph!
  "Like io!, but with a more appropriate name and message."
  [& body]
  `(io!
    "Graph modification in a transaction!"
    ~@body))

(defn lazy-sort
  ([xs]
     (lazy-sort compare xs))
  ([cmp xs]
     (lazy-seq
      (when (seq xs)
        (let [[p & r] xs
              smaller? #(< (cmp % p) 0)
              bigger-or-equal? (complement cmp)]
          (concat (lazy-sort (for [y r :when (smaller? y)] y))
                  [p]
                  (lazy-sort (for [y r :when (bigger-or-equal? y)] y))))))))


;;* Schema Access

(defprotocol Resolving
  "A protocol for resolving m1 classes and domains."
  (attributed-element-class [this] [this qname]
    "Returns this element's attributed element class or the m1 class with the
    given qname.")
  (m1class [this] [this qname]
    "Returns this element's m1 class or the m1 class with the given qname.")
  (domain [this] [this qname]
    "Returns the class (or M1 class for records) of the domain qname.")
  (schema [this]
    "Returns the schema of this element."))

(extend-protocol Resolving
  AttributedElement
  (attributed-element-class
    ([this]
       (.getAttributedElementClass this))
    ([this qname]
       (or (-> this .getSchema (.getAttributedElementClass (name qname)))
           (throw (RuntimeException.
                   ^String (format "No such attributed element class %" (name qname)))))))
  (m1class
    ([elem]
       (.getM1Class elem))
    ([elem qname]
       (let [aec (-> (.getSchema elem)
                     (.getAttributedElementClass (name qname)))]
            (if aec
              (.getM1Class aec)
              (throw (RuntimeException.
                      ^String (format "No such attributed element class %s" qname)))))))
  (domain [elem qname]
    (or (domain (-> (.getSchema elem)
                    (.getDomain (name qname))))
        (throw (RuntimeException.
                ^String (format "No such domain %s" (name qname))))))
  (schema [ae]
    (.getSchema ae))

  Domain
  (domain [this]
    (if (instance? RecordDomain this)
      (.getM1Class ^RecordDomain this)
      this))
  (schema [this]
    (.getSchema this))

  AttributedElementClass
  (attributed-element-class
    ([this]
       this)
    ([this qname]
       (-> this .getSchema (.getAttributedElementClass (name qname)))))
  (m1class
    ([aec]
       (.getM1Class aec))
    ([aec qname]
       (-> (.getSchema aec)
           (.getAttributedElementClass (name qname))
           (.getM1Class))))
  (domain [aec qname]
    (or (domain (-> (.getSchema aec)
                    (.getDomain (name qname))))
        (throw (RuntimeException.
                ^String (format "No such domain %s" (name qname))))))
  (schema [this]
    (.getSchema this))

  Schema
  (m1class
    ([s]
       (throw (RuntimeException. "A schema has no M1Class")))
    ([s qname]
       (-> s
           (.getAttributedElementClass (name qname))
           (.getM1Class))))
  (domain [s qname]
    (or (domain (.getDomain s (name qname)))
        (throw (RuntimeException. ^String (format "No such domain %s" (name qname))))))
  (schema [this]
    this))

(defn- type-matcher-1
  "Returns a matcher for elements Foo, !Foo, Foo!, !Foo!."
  [g c]
  (let [v (get-modifiers (name c))
        neg   (v 0)
        qname (v 1)
        exact (v 2)
        type  (m1class g qname)]
    (cond
     (and (not neg) (not exact)) (fn [x] (instance? type x))
     (and (not neg) exact)       (fn [x] (identical? type (m1class x)))
     (and neg       (not exact)) (fn [x] (not (instance? type x)))
     :default                    (fn [x] (not (identical? type (m1class x)))))))

(defn type-matcher
  "Returns a matcher for either nil, !Foo!, [Foo Bar! !Baz], [:and 'Foo 'Bar],
  or [:or 'Foo 'Bar].  In a collection spec, the first element may be one of
  the keywords :or (default), :and, or :xor."
  [g cls]
  (cond
   (nil? cls)   identity
   (fn? cls)    cls
   (qname? cls) (type-matcher-1 g cls)
   (coll? cls)  (if (seq cls)
                  (let [f (first cls)
                        [op r] (case f
                                 :and [every-pred (next cls)]
                                 :or  [some-fn    (next cls)]
                                 :xor [xor-fn     (next cls)]
                                 [some-fn    cls])
                        t-matchers (map #(type-matcher (schema g) %) r)]
                    (apply op t-matchers))
                  ;; Empty collection given: (), [], that's also ok
                  identity)
   :else (RuntimeException. "Don't know how to create a type matcher for" cls)))

(defn has-type?
  "Returns true, if attributed element ae has type cls.
cls can be given as anything type-matcher understands."
  [ae cls]
  ((type-matcher ae cls) ae))

(defprotocol QualifiedName
  "A protocol for qualified names."
  (qname [this]
    "Returns the qualified name of this named element's class or named element
    class."))

(extend-protocol QualifiedName
  AttributedElementClass
  (qname [aec]
    (.getQualifiedName aec))

  AttributedElement
  (qname [ae]
    (qname (.getAttributedElementClass ae))))

;;* Graph Access

(defn graph
  "Get the graph containing ge."
  [^GraphElement ge]
  (.getGraph ge))

;;** Access by ID

(defprotocol IDOps
  "Protocol for types having IDs."
  (id [this]
    "Returns this element's ID.")
  (vertex [this id]
    "Returns the vertex with the given ID.")
  (edge [this id]
    "Returns the edge with the given ID."))

(extend-protocol IDOps
  Graph
  (id [g]
      (.getId g))
  (vertex [g id]
	  (.getVertex g id))
  (edge [g id]
	(.getEdge g id))

  GraphElement
  (id [ge]
      (.getId ge))
  (edge [ge id]
	(-> (.getGraph ge)
	    (.getEdge id)))
  (vertex [ge id]
	  (-> (.getGraph ge)
	      (.getVertex id))))

;;** Edge functions

(defn alpha
  "Returns the start vertex of edge e."
  [^Edge e]
  (.getAlpha e))

(defn omega
  "Returns the end vertex of edge e."
  [^Edge e]
  (.getOmega e))

(defn this
  "Returns e's this vertex."
  [^Edge e]
  (.getThis e))

(defn that
  "Returns e's that vertex."
  [^Edge e]
  (.getThat e))

(defn normal-edge
  "Returns e's normal (forward-oriented) edge."
  [^Edge e]
  (.getNormalEdge e))

(defn reversed-edge
  "Returns e's reversed (backward-oriented) edge."
  [^Edge e]
  (.getReversedEdge e))

(defn normal-edge?
  "Returns true, if e is normal, false otherwise."
  [^Edge e]
  (.isNormal e))

;;** First, next elements

(defn first-vertex
  "Returns the first vertex of graph g accepted by type matcher tm."
  ([^Graph g]
     (first-vertex g identity))
  ([^Graph g tm]
     (loop [v (.getFirstVertex g)]
       (if (or (nil? v) (tm v))
         v
         (recur (.getNextVertex v))))))

(defn next-vertex
  "Returns the vertex following v in vseq accepted by type matcher tm."
  ([^Vertex v]
     (next-vertex v identity))
  ([^Vertex v tm]
     (loop [n (.getNextVertex v)]
       (if (or (nil? n) (tm n))
         n
         (recur (.getNextVertex n))))))

(defn first-edge
  "Returns the first edge of graph g accepted by type matcher tm."
  ([^Graph g]
     (first-edge g identity))
  ([^Graph g tm]
     (loop [e (.getFirstEdge g)]
       (if (or (nil? e) (tm e))
         e
         (recur (.getNextEdge e))))))

(defn next-edge
  "Returns the edge following e in eseq accepted by type matcher tm."
  ([^Edge e]
     (next-edge e identity))
  ([^Edge e tm]
     (loop [n (.getNextEdge e)]
       (if (or (nil? n) (tm n))
         n
         (recur (.getNextEdge n))))))

(defn direction-matcher
  "Returns a matcher function that accepts edges of direction dir.
  dir may be an EdgeDirection enum literal or keywords :in, :out, :inout.
  If dir is nil, then any direction is allowed (aka :inout)."
  [dir]
  ;; case does a constant time dispatch, so only use cond if the dir was not
  ;; given as keyword (or is nil).
  (case dir
    :out           normal-edge?
    :in            (complement normal-edge?)
    (:inout nil)   identity
    ;; too bad, not nil and not keyword...
    (cond
     (= dir EdgeDirection/OUT)   normal-edge?
     (= dir EdgeDirection/IN)    (complement normal-edge?)
     (= dir EdgeDirection/INOUT) identity
     :default (throw (RuntimeException.
                      ^String (format "Unknown direction %s" dir))))))

(defn first-inc
  "Returns the first incidence in iseq of v accepted by type/direction matcher tm/dm."
  ([^Vertex v]
     (first-inc v identity identity))
  ([^Vertex v tm]
     (first-inc v tm identity))
  ([^Vertex v tm dm]
     (loop [i (.getFirstIncidence v)]
       (if (or (nil? i) (and (dm i) (tm i)))
         i
         (recur (.getNextIncidence i))))))

(defn next-inc
  "Returns the incidence following e in the current vertex's iseq accepted by
  type/direction matcher tm/dm."
  ([^Edge e]
     (next-inc e identity identity))
  ([^Edge e tm]
     (next-inc e tm identity))
  ([^Edge e tm dm]
     (loop [i (.getNextIncidence e)]
       (if (or (nil? i) (and (dm i) (tm i)))
         i
         (recur (.getNextIncidence i))))))

;;** Value access (including attribute setting)

(defprotocol ClojureValues2JGraLabValues
  "Protocol for transforming clojure persistent collections/maps into
  equivalent pcollections."
  (clj-to-jg-value [coll]))

(extend-protocol ClojureValues2JGraLabValues
  clojure.lang.ISeq
  (clj-to-jg-value [coll]
    (-> (ArrayPVector/empty)
        (.plusAll ^Collection (map clj-to-jg-value coll))))

  clojure.lang.IPersistentVector
  (clj-to-jg-value [coll]
    (-> (ArrayPVector/empty)
        (.plusAll ^Collection (map clj-to-jg-value coll))))

  clojure.lang.IPersistentMap
  (clj-to-jg-value [coll]
    (reduce (fn [m [k v]] (.plus ^ArrayPMap m k v))
            (ArrayPMap/empty)
            (map (fn [k v] [k v])
                 (map clj-to-jg-value (keys coll))
                 (map clj-to-jg-value (vals coll)))))

  clojure.lang.IPersistentSet
  (clj-to-jg-value [coll]
    (-> (ArrayPSet/empty)
        (.plusAll ^Collection (map clj-to-jg-value coll))))

  Record (clj-to-jg-value [r] r)

  clojure.lang.Ratio (clj-to-jg-value [r] (double r))

  java.lang.Number (clj-to-jg-value [n] n)
  java.lang.String (clj-to-jg-value [s] s)
  java.lang.Boolean (clj-to-jg-value [b] b)

  java.lang.Enum (clj-to-jg-value [e] e)

  ;; PCollections stay as is
  org.pcollections.PCollection (clj-to-jg-value [coll] coll)
  org.pcollections.PMap (clj-to-jg-value [m] m)

  ;; nil stays nil/null
  nil (clj-to-jg-value [_] nil))

(defprotocol ValueAccess
  "Protocol for access to attribute values and record components."
  (value [this attr-or-comp]
    "Returns this element's attr-or-comp value.")
  (set-value! [this attr val]
    "Sets this element's attr value to val and returns this."))

(extend-protocol ValueAccess
  AttributedElement
  (value [ae attr]
    (.getAttribute ae (name attr)))

  (set-value! [ae attr val]
    (dograph! (doto ae (.setAttribute (name attr) (clj-to-jg-value val)))))

  Record
  (value [rec comp]
    (.getComponent rec (name comp))))

(defn record
  "Creates a jgralab record of type t (in the schema of element e) with
  component values as specified by map m.  The map m must specify all
  components, and be sure that if a component is of type Integer, then use
  Integer/valueOf."
  [^AttributedElement e t m]
  (-> e
      ^Schema (.getSchema)
      ^RecordDomain (.getDomain (name t))
      ^Class (.getM1Class)
      (.getConstructor (into-array Class [java.util.Map]))
      ;; zipmap to transform the keys into strings
      (.newInstance (into-array Object [(zipmap (map name (keys m))
                                                (vals m))]))))

(defn enum
  "Gets the enum constant c of type t in the schema of element e."
  [^AttributedElement e t c]
  (let [s (.getSchema e)
        cname (name c)
        enum-name (str (-> s .getPackagePrefix) "." t)
        enum-cls (Class/forName
                  enum-name
                  true
                  (M1ClassManager/instance (.getQualifiedName s)))
        consts (.getEnumConstants enum-cls)
        len (alength consts)]
    (loop [idx 0]
      (if (< idx len)
        (let [^Enum const (aget consts idx)]
          (if (= cname (.name const))
            const
            (recur (inc idx))))
        (throw (RuntimeException.
                ^String (format "No such constant %s in enum %s" c enum-name)))))))

;;* Modifications

;;** Creations

(defn create-graph
  "Creates a graph with id gid, vmax, and emax of the given schema class using impl type.
  vmax and emax default to 1000, impl to ImplementationType/STANDARD."
  ([schema gid]
     (create-graph schema gid 1000 1000 ImplementationType/STANDARD))
  ([schema gid vmax emax]
     (create-graph schema gid vmax emax ImplementationType/STANDARD))
  ([^Schema schema gid vmax emax ^ImplementationType impl]
     (let [^Method m (.getGraphCreateMethod schema impl)]
       (.invoke m nil (to-array [gid
                                 (Integer/valueOf (int vmax))
                                 (Integer/valueOf (int emax))])))))

(defn create-vertex!
  "Creates a new vertex in g of type cls.
cls may be a M1 class or a qualified name as string, symbol, or keyword."
  [^Graph g cls]
  (dograph! (.createVertex g (m1class g cls))))

(defn create-edge!
  "Creates a new edge of type cls starting at from and ending at to.
cls may be a M1 class or a qualified name as string, symbol, or keyword."
  [cls ^Vertex from ^Vertex to]
  (dograph! (.createEdge ^Graph (.getGraph from) (m1class from cls) from to)))

(defn set-alpha!
  "Sets the start vertex of e to v and returns e."
  [^Edge e ^Vertex v]
  (dograph! (doto e (.setAlpha v))))

(defn set-omega!
  "Sets the end vertex of e to v and returns e."
  [^Edge e ^Vertex v]
  (dograph! (doto e (.setOmega v))))

(defn set-this!
  "Sets the this vertex of i to v and returns i."
  [^Edge i ^Vertex v]
  (dograph! (doto i (.setThis v))))

(defn set-that!
  "Sets the that vertex of i to v and returns i."
  [^Edge i ^Vertex v]
  (dograph! (doto i (.setThat v))))

;;** Deletions

(defprotocol Delete
  "A protocol for deleting elements."
  (internal-delete [this]
    "Deletes this vertex or edge."))

(extend-protocol Delete
  Vertex (internal-delete [v] (dograph! (.delete v)))
  Edge   (internal-delete [e] (dograph! (.delete e))))

(alter-meta! (var internal-delete) assoc :private true)

(defn delete!
  "Deletes the graph elems."
  [& elems]
  (doseq [ge elems]
    (internal-delete ^GraphElement ge)))

;;** Relinking edges

(defn relink!
  "Relink all incidences of vertex from to vertex to and return from.
  The incidences can be restricted by dir, and cls."
  ([from to]
     (relink! from to identity identity))
  ([from to cls]
     (relink! from to cls identity))
  ([from to cls dir]
     (let [tm (type-matcher from cls)
           dm (direction-matcher dir)]
       (loop [inc (first-inc from tm dm)]
         (when inc
           (set-this! inc to)
           (recur (first-inc from tm dm)))))
     from))

;;* Serialization/Deserialization

(def ^:dynamic
  *serialization-bindings* nil)

(let [sb-qname     (subs (str (var *serialization-bindings*)) 2)
      vertex-qname (subs (str (var vertex)) 2)
      edge-qname   (subs (str (var edge)) 2)]
  (defmethod print-dup Vertex [v out]
    (.write ^java.io.Writer out
            (format "#=(%s #=(%s \"%s\") %s)"
                    vertex-qname sb-qname (id (graph v)) (id v))))

  (defmethod print-dup Edge [e out]
    (.write ^java.io.Writer out
            (format "#=(%s #=(%s \"%s\") %s)"
                    edge-qname sb-qname (id (graph e)) (id e))))

  (defmethod print-dup Graph [g out]
    (.write ^java.io.Writer out
            (format "#=(%s \"%s\")"
                    sb-qname (id g)))))

(defn tg-pr-str
  "Prints arbitrary clojure data structures plus vertices and edges to a string
  that can be read back using tg-read-str."
  [obj]
  (binding [*print-dup* true]
    (pr-str obj)))

(defn tg-read-str
  "Reads str and returns the object represented by str.
  If the object contains vertices or edges, the graphs holding them have to be
  provided as gs."
  [str & gs]
  (binding [*serialization-bindings* (into {} (map (fn [g] [(id g) g])
                                                   gs))]
    (read-string str)))

(defn tg-spit
  "Like spit but ensure that vertices and edges are printed in a readable manner."
  [f obj]
  (spit f (tg-pr-str obj)))

(defn tg-slurp
  "Reads an object from the file f.
  If that object contains vertices and edges, then their hosting graphs have to
  be provided as gs."
  [f & gs]
  (apply tg-read-str (slurp f) gs))
