(ns de.uni-koblenz.ist.funtg.trace-graph
  (:use de.uni-koblenz.ist.funtg.core)
  (:use de.uni-koblenz.ist.funtg.funql)
  (:import
   (de.uni_koblenz.jgralab Graph Vertex Edge AttributedElement GraphElement)
   (de.uni_koblenz.jgralab.impl InternalVertex InternalEdge)
   (de.uni_koblenz.jgralab.schema Attribute RecordDomain GraphClass
                                  GraphElementClass AttributedElementClass VertexClass EdgeClass
                                  AggregationKind)))


(deftype TraceVertex [^{:unsynchronized-mutable true} graph
                      ^{:unsynchronized-mutable true} attributes
                      ^{:unsynchronized-mutable true} first-inc
                      ^{:unsynchronized-mutable true} next-vertex]
  GraphElement
  (getId [this] -1)
  (getGraph [this] graph)
  (isValid [this] (not (nil? graph)))
  (delete [this]
    (.deleteVertex graph this)
    (set! graph nil))
  AttributedElement
  (getAttribute [this name]
    (attributes name))
  (setAttribute [this name val]
    (set! attributes (assoc attributes name val)))
  Vertex
  (getNextVertex [this]
    next-vertex)
  InternalVertex
  (setNextVertex [this n]
    (set! next-vertex n)))

(deftype TraceEdge [graph alpha omega
                    ^{:unsynchronized-mutable true} next-inc
                    ^{:unsynchronized-mutable true} next-edge
                    reversed-edge]
  Edge
  (getAlpha [this] alpha)
  (getOmega [this] omega)
  (getThis  [this] alpha)
  (getThat  [this] omega)
  (getNormalEdge   [this] this)
  (getReversedEdge [this] reversed-edge)
  (getNextIncidence [this] next-inc)
  (delete [this]
    (.deleteEdge graph this))
  InternalEdge
  (setNextIncidenceInternal [this n]
    (set! next-inc n))
  (setNextEdgeInGraph [this n]
    (set! next-edge n)))

(deftype ReversedTraceEdge [graph alpha omega
                            ^{:unsynchronized-mutable true} next-inc]
  Edge
  (getAlpha [this] alpha)
  (getOmega [this] omega)
  (getThis  [this] omega)
  (getThat  [this] alpha)
  (getNormalEdge   [this]
    (loop [es (eseq graph)]
      (if (= (.getReversedEdge (first es)) this)
        (first es)
        (recur (rest es)))))
  (getReversedEdge [this] this)
  (getNextIncidence [this] next-inc)
  (delete [this]
    (when-let [n (.getNormalEdge this)]
      (.deleteEdge graph n)))
  InternalEdge
  (setNextIncidenceInternal [this n]
    (set! next-inc n)))

(deftype TraceGraph [^{:unsynchronized-mutable true} attributes
                     ^{:unsynchronized-mutable true} vertices
                     ^{:unsynchronized-mutable true} edges]
  AttributedElement
  (getAttribute [this name]
    (attributes name))
  (setAttribute [this name val]
    (set! attributes (assoc attributes name val)))
  Graph
  (getFirstVertex [this] (first vertices))
  (getFirstEdge [this] (first edges))
  (createVertex [this cls]
    (let [nv (TraceVertex. this {} nil nil)]
      (set! vertices (conj vertices nv))
      (.setNextVertex (last (vseq this)) nv)
      nv))
  (createEdge [this cls a o]
    (let [la (last (iseq a))
          lo (last (iseq o))
          re (ReversedTraceEdge. this a o nil)
          e  (TraceEdge. this a o nil nil re)]
      (.setNextIncidenceInternal la e)
      (.setNextIncidenceInternal lo re)
      (set! edges (conj edges e))
      (.setNextEdgeInGraph (last (eseq this)) e)
      e))
  (deleteVertex [this v]
    (let [n (.getNextVertex v)
          p (loop [xs (vertices)]
              (when (seq xs)
                (if (= (.getNextVertex (first xs)) v)
                  (first xs)
                  (recur (rest xs)))))]
      (when p
        (.setNextVertex p n))
      (doseq [e (iseq v)]
        (.delete e))
      (set! vertices (remove #(= v %) vertices))))
  (deleteEdge [this e]
    (let [n (.getNextIncidence e)
          p (loop [is (iseq (this e))]
              (when (seq is)
                (if (= (.getNextIncidence (first is)) e)
                  (first is)
                  (recur (rest is)))))]
      (when n
        (.setNextIncidence p n)))
    (set! edges (remove #(= e %) edges))))
