(ns life-cqi.ac
  "Access Control API"
  (:require [async-error.core :refer [<??]]
            [clojure.core.async :refer [<! go]]
            [clojure.spec :as s]
            [hap-client.core :as hap]
            [life-cqi.spec :as spec]
            [taoensso.timbre :refer [error trace]]))

(defn- fetch-query* [service-uri key]
  (get-in (<?? (hap/fetch service-uri)) [:queries key]))

(def ^:private fetch-query
  "Returns the possibly cached HAP query with the given key."
  (memoize fetch-query*))

(defn- coerce-response
  "Returns permission denied on any errors."
  [resp]
  (if (instance? Throwable resp)
    (do (error {:error {:type :error-response :msg (.getMessage resp)}})
        {:cqi.ac/permission :denied})
    (let [{{permission :permission scopes :scopes} :data} resp]
      (cond
        (and permission scopes)
        {:cqi.ac/permission :granted-scoped
         :cqi.ac/scopes scopes}
        permission
        {:cqi.ac/permission :granted-unscoped}
        :else
        {:cqi.ac/permission :denied}))))

(defn- operation [key]
  (if-let [namespace (namespace key)]
    (str namespace "/" (name key))
    (name key)))

(defn- params [{:keys [cqi.parser/dispatch-key cqi.ac/scope-ns]}]
  (cond-> {:operation (operation dispatch-key)}
          scope-ns (assoc :scope-ns scope-ns)))

(defn- fetch-permission-by-role [service-uri request roles]
  (let [params (assoc (params request) :roles roles)]
    (go
      (let [response (-> (fetch-query service-uri :ac/permissions-by-roles)
                         (hap/query params)
                         (<!)
                         (coerce-response))]
        (trace {:action :fetch-permission-by-role :params params
                :response response})
        response))))

(defn- params-by-user
  [{:keys [cqi.auth/sub cqi.ac/choosen-role] :as request}]
  (cond-> (assoc (params request) :user sub)
          choosen-role (assoc :role choosen-role)))

(defn- fetch-permission-by-user [service-uri request]
  (let [params (params-by-user request)]
    (go
      (let [response (-> (fetch-query service-uri :ac/permissions)
                         (hap/query params)
                         (<!)
                         (coerce-response))]
        (trace {:action :fetch-permission-by-user :params params
                :response response})
        response))))

(defn- roles
  "Returns the roles of a role-based request.

  If there is a choosen role and it exists in roles, it will be returnd as the
  only role. If it's not contained in roles, nothing will be returned."
  [{:keys [cqi.ac/choosen-role] :as request}]
  (when-let [roles (:cqi.auth/roles request)]
    (if choosen-role
      (when (some #{choosen-role} roles)
        [choosen-role])
      roles)))

(s/fdef fetch-permission
  :args (s/cat :service-uri string?
               :request :cqi.ac/permission-request))

(defn fetch-permission
  "Fetches permission information from AC service with URI using the request.

  Returns a channel conveying the :cqi.ac/permission-response."
  [service-uri request]
  (if-let [roles (roles request)]
    (fetch-permission-by-role service-uri request roles)
    (fetch-permission-by-user service-uri request)))

(s/fdef permission
  :args (s/cat :ac :cqi.ac/service
               :request :cqi.ac/permission-request))

(defn permission
  "Fetches permission information.

  Returns a channel conveying the :cqi.ac/permission-response."
  [service request]
  (spec/-permission service request))

(defrecord DefaultAC [service-uri]
  spec/AccessControl
  (-permission [_ request]
    (fetch-permission service-uri request)))

(s/fdef default-ac
  :args (s/cat :service-uri string?))

(defn default-ac [service-uri]
  (->DefaultAC service-uri))
