(ns life-cqi.core
  (:require [clojure.spec :as s]
            [datomic.api :as d]
            [io.pedestal.interceptor.chain :as chain]
            [life-cqi.spec]
            [life-cqi.interceptor.queries :refer [queries]]
            [om.next.impl.parser :as om]
            [taoensso.timbre :refer [trace warn]]))

(defn- clear-chain
  "Clears all interceptor chain keys from context."
  [context]
  (-> context
      (dissoc ::chain/queue)
      (dissoc ::chain/stack)
      (dissoc ::chain/execution-id)))

(defn deliver-result
  "Puts the result into the context under the :cqi/result key."
  [context result]
  (assoc context :cqi/result result))

(defn- merge-routes [interceptors]
  (apply merge (map #(if (delay? %) @% %) (map :cqi/routes interceptors))))

(s/fdef parser
  :args (s/cat :context (s/keys :req [:cqi/query]))
  :ret (s/keys :req [:cqi/result]))

(defn- parser
  "Takes a query and an optional root from context and executes the query
  global or relative to the root.

  Delivers the result as map of query keys to query results. Uses routes from
  context to find interceptors executing individual query expressions. The
  result is at least an empty map.

  Example
    * :cqi/root - {:id 1 :name \"foo\"}
    * :cqi/query - [:id :name]
    * prop interceptors on :id and :name
    * result: {:id 1 :name \"foo\"}"
  [{:keys [cqi/routes cqi/query cqi/root] :as context}]
  (trace {:action :enter-parser :query (pr-str query) :root (pr-str root)})

  ;; We reduce over the query which is a seq of query expressions. We build a
  ;; result as map of query expression keys to query results.
  (let [routes (if (delay? routes) @routes routes)
        sub-context
        (-> (clear-chain context)
            (dissoc :cqi/query)
            (dissoc :cqi.parser/params)
            (dissoc :cqi.parser/query-root))
        step
        (fn parser-step [ret expr]
          (trace {:action :execute-parser-step :expr (pr-str expr) :root (pr-str root)})
          (let [{next-query :query :keys [key dispatch-key params] :as ast}
                (om/expr->ast expr)]
            (if-let [interceptors (get routes dispatch-key)]
              (let [query (if (= '... next-query) query next-query)
                    context
                    (cond-> (assoc sub-context
                              :cqi.parser/dispatch-key dispatch-key
                              :cqi/routes (merge-routes interceptors)
                              :cqi/parent-routes routes)
                            params (assoc :cqi.parser/params params)
                            (vector? query) (assoc :cqi/query query)
                            (map? query) (assoc :cqi/union-query query)
                            (vector? key) (assoc :cqi.parser/query-root key))]
                (if-let [result (:cqi/result (chain/execute context interceptors))]
                  (assoc ret key result)
                  ret))
              (do (warn {:error :missing-route :dispatch-key dispatch-key})
                  ret))))]
    (deliver-result context (reduce step {} query))))

(s/fdef join
  :args (s/cat :context map?
               :xform (s/? fn?)
               :root some?))

(defn join
  "Executes current query from context as join over root."
  [context root]
  (parser (assoc context :cqi/root root)))

(defn- join-step!
  [{:keys [cqi/query cqi/union-query cqi/union-key-fn]
    :or {union-key-fn :cqi/union-key}
    :as context}]
  (cond
    query
    (fn
      ([ret]
       ret)
      ([ret root]
       (let [context (parser (assoc context :cqi/root root))]
         (conj! ret (:cqi/result context)))))
    union-query
    (fn
      ([ret]
       ret)
      ([ret root]
       (if-let [query (get union-query (union-key-fn root))]
         (let [context (parser (assoc context :cqi/query query :cqi/root root))]
           (conj! ret (:cqi/result context)))
         (do (warn {:error :missing-union-entry :union-query (pr-str union-query) :root (pr-str root)})
             ret))))
    :else
    (throw (ex-info "Missing query in context while join." {:context context}))))

(s/fdef join-many
  :args (s/cat :context map?
               :xform (s/? fn?)
               :roots seqable?))

(defn join-many
  "Executes current query from context as join over roots.

  An optional xform can be supplied which will be executed before the join. The
  xform can be used to filter the roots."
  ([context roots]
   (->> (persistent! (reduce (join-step! context) (transient []) roots))
        (deliver-result context)))
  ([context xform roots]
   (->> (persistent! (transduce xform (join-step! context) (transient []) roots))
        (deliver-result context))))

(s/fdef pull
  :args (s/cat :context (s/keys :req [:cqi/query])
               :xform (s/? fn?)
               :roots seqable?))

(defn pull
  "Like join but uses a direct Datomic pull. Roots have to be Datomic entities.

  No interceptors will be used."
  ([{:keys [cqi/query] :as context} roots]
   (pull context (map identity) roots))
  ([{:keys [cqi/query] :as context} xform roots]
   (->> (if (seq roots)
          (d/pull-many (d/entity-db (first roots)) query
                       (into [] (comp (map :db/id) xform) roots))
          [])
        (deliver-result context))))

(defn prop
  "An interceptor which returns the value of the prop with key or the current
  :cqi.parser/dispatch-key if not given."
  ([]
   {:name :cqi/prop
    :cqi.query/type :prop
    :enter
    (fn enter-prop [{:keys [cqi/root cqi.parser/dispatch-key] :as context}]
      (if-let [val (dispatch-key root)]
        (deliver-result context val)
        context))})
  ([key]
   {:name key
    :cqi.query/type :prop
    :enter
    (fn enter-prop [{:keys [cqi/root] :as context}]
      (if-let [val (key root)]
        (deliver-result context val)
        context))}))

(defn const
  "An interceptor which returns the constant value v."
  [v]
  {:name :const
   :cqi.query/type :prop
   :enter
   (fn enter-const [context]
     (deliver-result context v))})

(s/fdef execute
  :args (s/cat :context (s/keys :opt [:cqi/routes])
               :query :cql/query-root))

(defn execute
  "Executes query on context. Returns the result.

  The context should contain :cqi/routes."
  [context query]
  (-> (assoc context :cqi/query query)
      (update :cqi/routes assoc :cqi/queries [queries])
      (parser)
      (:cqi/result)))
