(ns life-cqi.interceptor.ac
  (:require [clojure.core.async :refer [<!!]]
            [clojure.core.cache :as cache]
            [clojure.spec :as s]
            [io.pedestal.interceptor.chain :as chain]
            [life-cqi.ac :as ac]
            [life-cqi.spec]
            [taoensso.timbre :refer [trace]])
  (:import [clojure.lang IAtom]))

(defn- ac-request [{:keys [cqi.auth/identity]} key scope-ns]
  (cond-> (assoc identity :cqi.parser/dispatch-key key)
          scope-ns (assoc :cqi.ac/scope-ns scope-ns)))

(s/fdef update-cache!
  :args (s/cat :cache #(instance? IAtom %)
               :service :cqi.ac/service
               :request :cqi.ac/permission-request))

(defn- update-cache! [cache service request]
  (if (cache/has? @cache request)
    (swap! cache cache/hit request)
    (swap! cache cache/miss request (<!! (ac/permission service request)))))

(defn- permission
  "Asks for permission for the identity in context under key and scope-ns.

  Uses the AC cache in context to minimize network fetches."
  [{:keys [:cqi.ac/service :cqi.ac/cache] :as context} key scope-ns]
  (let [request (ac-request context key scope-ns)
        response (get (update-cache! cache service request) request)]
    {:permission (:cqi.ac/permission response)
     :scopes (:cqi.ac/scopes response)}))

(s/def ::context
  (s/keys :req [:cqi.ac/service :cqi.auth/identity :cqi.ac/cache]))

(defn ac
  "Access control interceptor which takes a scope namespace and attaches
  :cqi.ac/permission and :cqi.ac/scopes to the context.

  Requires :cqi.ac/service, :cqi.auth/identity and :cqi.ac/cache to be present
  in the context.

  Terminates interceptor chain if permission is denied."
  [scope-ns]
  {:name :life-cqi.interceptor/ac
   :enter
   (fn [{:keys [cqi.parser/dispatch-key]
         {:keys [cqi.auth/sub cqi.auth/roles]} :cqi.auth/identity
         :as context}]
     (s/assert ::context context)
     (let [{:keys [permission scopes]} (permission context dispatch-key scope-ns)]
       (trace {:action :enter-ac-interceptor :dispatch-key dispatch-key
               :scope-ns scope-ns :sub sub :roles roles :permission permission
               :scopes scopes})
       (if (= :denied permission)
         (chain/terminate context)
         (cond-> (assoc context :cqi.ac/permission permission)
                 (nil? scopes) (dissoc :cqi.ac/scopes)
                 scopes (assoc :cqi.ac/scopes scopes)))))})
