(ns life-cqi.spec
  (:require [clojure.spec :as s]))

;; ---- Command and Query Language --------------------------------------------

(s/def :cql/query-root
  (s/coll-of :cql/query-expr :kind vector?))

(s/def :cql/query-expr
  (s/or :plain :cql/plain-query-expr :param :cql/param-expr))

(s/def :cql/plain-query-expr
  (s/or :prop keyword? :ident :cql/ident-expr :join :cql/join-expr))

(s/def :cql/ident-expr
  (s/tuple keyword? any?))

(s/def :cql/param-expr
  (s/cat :expr (s/alt :query :cql/plain-query-expr :command symbol?)
         :params :cql/param-map-expr))

(s/def :cql/param-map-expr
  (s/map-of keyword? any?))

(s/def :cql/join-expr
  (s/map-of
    (s/or :prop keyword? :ident :cql/ident-expr)
    (s/or :root :cql/query-root :union :cql/union-expr :recur :cql/recur-expr)
    :count 1))

(s/def :cql/union-expr
  (s/map-of keyword? :cql/query-root))

(s/def :cql/recur-expr
  (s/or :inf #{'...} :depth nat-int?))

;; ---- Context ---------------------------------------------------------------

(s/def :cqi/union-key-fn
  ifn?)

;; ---- Parser ----------------------------------------------------------------

;; can be a delay
(s/def :cqi/routes
  (s/map-of keyword? (s/coll-of some? :kind vector?)))

(s/def :cqi/parent-routes
  :cqi/routes)

;; The entity on which the current query is based
(s/def :cqi/root
  some?)

(s/def :cqi/union-key
  keyword?)

(s/def :cqi/query
  :cql/query-root)

(s/def :cqi/union-query
  :cql/union-expr)

(s/def :cqi/result
  any?)

(s/def :cqi.parser/query-root
  vector?)

(s/def :cqi.parser/dispatch-key
  keyword?)

(s/def :cqi.parser/params
  (s/nilable map?))

;; ---- Query Discovery -------------------------------------------------------

(s/def :cqi.query/key keyword?)

;; A query can be either a prop (a scalar) or a join to another entity or list
;; of entities (see: cardinality).
(s/def :cqi.query/type
  #{:prop :join})

;; Cardinality many queries return a list were cardinality one queries return a
;; single item.
(s/def :cqi.query/cardinality
  #{:one :many})

(s/def :cqi.query/spec
  some?)

(s/def :cqi.query/param-spec
  some?)

(s/def :cqi/queries
  (s/coll-of (s/keys :req [:cqi.query/key :cqi.query/type
                           :cqi.query/cardinality]
                     :opt [:cqi.query/spec])))

;; ---- Auth ------------------------------------------------------------------

;; Subject Identifier
(s/def :cqi.auth/sub
  string?)

;; Time at which the token was issued. This time can be used to invalidate
;; caches if a new token is used.
(s/def :cqi.auth/iat
  inst?)

;; A set of roles which can be used to determine permissions instead of using
;; the subject identifier directly.
(s/def :cqi.auth/roles
  (s/coll-of string? :kind set?))

;; The identity (token) of a subject. It's used as cache key and parts of it are
;; used in permission requests.
(s/def :cqi.auth/identity
  (s/keys :req [:cqi.auth/sub]
          :opt [:cqi.auth/iat :cqi.auth/roles]))

;; ---- Access Control --------------------------------------------------------

(defprotocol AccessControl
  (-permission [this request]))

(s/def :cqi.ac/service
  #(satisfies? AccessControl %))

;; A role the user has choosen to restrict his permissions.
(s/def :cqi.ac/choosen-role
  string?)

(s/def :cqi.ac/scope-ns
  string?)

;; A permission request to the access control service.
;; If roles are present, the subject (user) is not used in the request.
;; The caller has to ensure that the user actually has the given roles.
(s/def :cqi.ac/permission-request
  (s/keys :req [(or :cqi.auth/sub :cqi.auth/roles) :cqi.parser/dispatch-key]
          :opt [:cqi.ac/choosen-role :cqi.ac/scope-ns]))

(s/def :cqi.ac/permission
  #{:granted-unscoped :granted-scoped :denied})

(s/def :cqi.ac/scopes
  (s/coll-of string? :kind set?))

;; A permission response from the access control service.
(s/def :cqi.ac/permission-response
  (s/keys :req [:cqi.ac/permission]
          :opt [:cqi.ac/scopes]))

(s/def :cqi.ac/cache
  some?)
