(ns spectator.core
  (:require [clojure.spec.alpha :as s]
            [spectator.generators :as tgen])
  (:import (java.time LocalDate)))

;; Some additional predicates
(defn string-1?
  "Returns true if x is a non-empty string"
  [str]
  (and (string? str)
       (not (empty? str))))

(defn min-length?
  [n x]
  (<= n (count x)))

(defn local-date?
  "Returns true if x is a instance of java.time.LocalDate"
  {:static true}
  [x] (instance? LocalDate x))

(defn future-local-date?
  "Returns true if x is a local-date in the future (today is considered part of future)"
  [d] (and
        (local-date? d)
        (let [today (LocalDate/now)]
          (or (.isAfter d today)
              (= d today)))))

(defn spec?
  [x]
  (if (s/spec? x) true false))

;; Specs (with custom generators) for above predicates
(s/def ::string-1           (s/spec string-1? :gen tgen/string-1))
(s/def ::local-date         (s/spec local-date? :gen tgen/local-date))
(s/def ::future-local-date  (s/spec future-local-date? :gen tgen/future-local-date))

(s/def :keys/keys-vector    (s/coll-of keyword? :kind vector?))
(s/def :keys/req            :keys/keys-vector)
(s/def :keys/req-un         :keys/keys-vector)
(s/def :keys/opt            :keys/keys-vector)
(s/def :keys/opt-un         :keys/keys-vector)
(s/def ::keys-form          (s/cat :type #{'clojure.spec.alpha/keys 'keys 's/keys}
                                   :options (s/keys* :opt-un [:keys/req :keys/req-un
                                                              :keys/opt :keys/opt-un])))
(s/def ::merge-form         (s/cat :type #{'clojure.spec.alpha/merge 'merge}
                                   :keys (s/+ any?)))

(s/def :every/kind          any?)
(s/def :every/count         nat-int?)
(s/def :every/min-count     nat-int?)
(s/def :every/max-count     nat-int?)
(s/def :every/distinct      boolean?)
;(s/def :every/into #{'[] '() '{} '#{}})
(s/def ::every-form         (s/cat :type #{'clojure.spec.alpha/every 'every 'clojure.spec.alpha/coll-of}
                                   :spec any? #_(s/or :spec-form spec?
                                                         :symbol symbol?
                                                         :kw keyword?)
                                   :options (s/keys* :opt-un [:every/kind :every/count :every/min-count
                                                              :every/max-count :every/distinct ])))

;(s/def ::and-form (s/cat :type #{'clojure.spec.alpha/and 'and} ))

#_(s/def ::spec-form  (s/or :keys-form ::keys-form
                          :every-form ::every-form
                          :kw keyword?
                          :spec s/spec?))

(defn remove-namespace [kw]
  (keyword (first (clojure.string/split (name kw) #"/"))))

(defn check-un [keys-type spec-name]
  (if (clojure.string/ends-with? (name keys-type) "un")
    (remove-namespace spec-name)
    spec-name))

(defn keys-spec-name [keys-type spec-name]
  (let [namespace-checked-name (check-un keys-type spec-name)]
    (if (clojure.string/starts-with? (name keys-type) "opt")
      [namespace-checked-name :opt]
      namespace-checked-name)))

(defn combine-merge-form
  [merge-form]
  (->> (s/form ::customer)
       rest
       (map #(s/conform :spectator.core/keys-form (s/form %)))
       (map #(-> % :options))
       (apply (partial merge-with concat))))

(def default-formats
  {:keys-spec-name keys-spec-name})


(s/fdef keys-spec?
        :args (s/cat :spec-form ::keys-form)
        :ret  boolean?)

(defn keys-spec? [spec-desc]
  (s/valid? ::keys-form spec-desc))

(defn set-spec? [spec-desc]
  (set? spec-desc))

(defn merge-spec? [spec-desc]
  (s/valid? ::merge-form spec-desc))

(s/fdef coll-spec?
        :args (s/cat :spec-form ::every-form)
        :ret  boolean?)

(defn coll-spec? [spec-form]
  (s/valid? ::every-form spec-form))

(s/fdef vector-spec?
        :args (s/cat :spec-form ::every-form)
        :ret  boolean?)

(defn vector-spec? [spec-desc]
  (-> (s/conform ::every-form spec-desc)
      :options
      :clojure.spec.alpha/kind-form
      (= 'clojure.core/vector?)))

(defn and-spec? [spec-desc]
  (and (seq? spec-desc)
       (= 'clojure.spec.alpha/and (first spec-desc))))

;(defn pred->nat-lang
;  ([pred]
;   (pred->nat-lang pred (partial get-doc-string true)))
;  ([pred format-fn]
;   (cond
;     (seq? pred) (clojure.string/join " " (map (fn [x] (pred->nat-lang x)) pred))
;     (symbol? pred) (format-fn pred)
;     :else pred)))


(declare expand-spec)

(defn expand-keys-form
  [keys-form format-fn]
  (->> (s/conform ::keys-form keys-form)
       :options
       (map (fn [e]
              (let [keys-type (key e)
                    req-spec-names (val e)]
                (merge
                  (apply merge
                         (map (fn [child-spec-name]
                                (array-map (keys-spec-name keys-type child-spec-name)
                                           (expand-spec child-spec-name format-fn)))
                              req-spec-names))))))
       (apply merge)))

`
#_(s/fdef expand-spec
        :args (s/cat :spec ::spec-form :format-fn (s/? fn?))
        :ret any?)

(defn expand-spec
  "Expands a spec, i.e. returns the spec, together with any children specs, recursively.
  'Keys' specs are returned as maps, and 'every' specs are returned as lists or vectors."
  ([spec]
    (expand-spec spec identity))
  ([spec format-fn]
   (let [spec-form (if (or (s/spec? spec)
                           (keyword? spec))
                     (s/describe spec)
                     spec)]
     (cond
       (s/valid? ::keys-form spec-form)  (expand-keys-form spec-form format-fn)
       (s/valid? ::every-form spec-form) (expand-every-form spec-form format-fn)
       :else                             (format-fn spec-form)))))

(defn path->str
  ([path]
   (path->str path " > "))
  ([path separator]
   (->> (clojure.string/join separator path)
        (str separator))))
