(ns spectator.swagger
  (:require [clojure.spec.alpha     :as s]
            [clojure.spec.gen.alpha :as gen]
            #_[cheshire.core    :as cheshire]
            [clj-yaml.core    :as yaml]
            #_[com.rpl.specter :as specter]
            [spectator.core   :as sp]))

(s/def ::swagger-root                           (s/keys :req-un [::swagger
                                                                 ::info
                                                                 ::paths]
                                                        :opt-un [::host
                                                                 ::basePath
                                                                 ::schemes
                                                                 ::consumes
                                                                 ::produces
                                                                 ;::definitions
                                                                 ;::parameters
                                                                 ;::responses
                                                                 ;::securityDefinitions
                                                                 ;::security
                                                                 ::tags
                                                                 ;::externalDocs
                                                                 ]))

(s/def ::swagger                                (s/spec (fn [x] (= x "2.0"))
                                                        :gen (fn []  (s/gen #{"2.0"}))))
(s/def :spectator.swagger.info.contact/email    string?)

(s/def ::info                                   (s/keys :req-un [:spectator.swagger.info/title
                                                                 :spectator.swagger.info/version]
                                                        :opt-un [::description
                                                                 :spectator.swagger.info/termsOfService
                                                                 :spectator.swagger.info/contact
                                                                 :spectator.swagger.info/license]))

(s/def :spectator.swagger.info/title            string?)
(s/def :spectator.swagger.info/version          string?)
(s/def :spectator.swagger.info/description      string?)
(s/def :spectator.swagger.info/termsOfService   string?)
(s/def :spectator.swagger.info/contact          (s/keys :opt-un [::name
                                                                 ::url
                                                                 ::email]))
(s/def :spectator.swagger.info/license          (s/keys :req-un [::name
                                                                 ::url]))

(s/def ::host                                   string?)
(s/def ::basePath                               string?)
(s/def ::scheme                                 #{:http :https :ws :wss})
(s/def ::schemes                                (s/coll-of ::scheme))
(s/def ::mime-type                              string?)
(s/def ::consumes                               (s/coll-of ::mime-type))
(s/def ::produces                               (s/coll-of ::mime-type))
(s/def :spectator.swagger.tag/name              keyword?)
(s/def ::tag                                    (s/keys :req-un [:spectator.swagger.tag/name]
                                                        :opt-un [::description
                                                                 ::externalDocs]))
(s/def ::tags                                   (s/coll-of ::tag))

(s/def ::paths                                  (s/map-of  ::path ::path-item))
(s/def ::path                                   (s/and string?
                                                       (fn [x] (clojure.string/starts-with? x "/"))))

(s/def ::path-item                              (s/map-of :spectator.swagger.path/method ::operation))

(s/def :spectator.swagger.path/method           #{:get :put :post :delete :options :head :patch})

(s/def ::operation                              (s/keys :req-un [:spectator.swagger.operation/responses]
                                                        :opt-un [:spectator.swagger.operation/tags
                                                                 :spectator.swagger.operation/summary
                                                                 ::description
                                                                 ::externalDocs
                                                                 :spectator.swagger.operation/operationId
                                                                 :spectator.swagger.operation/consumes
                                                                 :spectator.swagger.operation/produces
                                                                 :spectator.swagger.operation/parameters
                                                                 ::schemes
                                                                 :spectator.swagger.operation/deprecated
                                                                 :spectator.swagger.operation/security]))

(s/def :spectator.swagger.operation/responses   (s/map-of ::http-status-codes :spectator.swagger.operation/response))
(s/def :spectator.swagger.operation/response    (s/keys :req-un [::description]
                                                        :opt-un [
                                                                 ;::schema
                                                                 ::headers
                                                                 ::examples]))
(s/def :spectator.swagger.operation/tags        (s/coll-of :spectator.swagger.tag/name))
(s/def :spectator.swagger.operation/summary     string?)
(s/def :spectator.swagger.operation/operationId string?)
(s/def :spectator.swagger.operation/consumes    (s/coll-of ::mime-type))
(s/def :spectator.swagger.operation/produces    (s/coll-of ::mime-type))
(s/def :spectator.swagger.operation/parameters  (s/coll-of :spectator.swagger.operation/parameter))
(s/def :spectator.swagger.operation/parameter   (s/keys :req-un [:spectator.swagger.operation.parameter/name
                                                                 :spectator.swagger.operation.parameter/in]
                                                        :opt-un [::description
                                                                 :spectator.swagger.operation.parameter/required
                                                                 :spectator.swagger.operation.parameter/schema
                                                                 :spectator.swagger.operation.parameter/type
                                                                 :spectator.swagger.operation.parameter/format
                                                                 :spectator.swagger.operation.parameter/allowEmptyValue
                                                                 :spectator.swagger.operation.parameter/items
                                                                 :spectator.swagger.operation.parameter/collectionFormat
                                                                 :spectator.swagger.operation.parameter/default
                                                                 ::maximum
                                                                 ::exclusiveMaximum
                                                                 ::minimum
                                                                 ::exclusiveMinimum
                                                                 ::maxLength
                                                                 ::minLength
                                                                 ::pattern
                                                                 ::maxItems
                                                                 ::minItems
                                                                 ::uniqueItems
                                                                 ::enum
                                                                 ::multipleOf]))

(s/def :spectator.swagger.operation.parameter/name      string?)
(s/def :spectator.swagger.operation.parameter/in        string?)
(s/def :spectator.swagger.operation.parameter/required  boolean?)
;(s/def :spectator.swagger.operation.parameter/schema    ::schema)
(s/def :spectator.swagger.operation.parameter/type      #{:string :number :integer :boolean :array :file})
(s/def :spectator.swagger.operation.parameter/format    #{:int32 :int64 :float :double :byte :binary :date :date-time :password})
(s/def :spectator.swagger.operation.parameter/allowEmptyValue  boolean?)
(s/def :spectator.swagger.operation.parameter/items     (s/keys :req-un [::description]
                                                                :opt-un [::schema
                                                                         ::headers
                                                                         ::examples]))

(s/def :spectator.swagger.operation.parameter/collectionFormat string?)
(s/def :spectator.swagger.operation.parameter/default   any?)

(s/def :spectator.swagger.operation/security    (s/coll-of map?))



(s/def ::name                                   string?)
(s/def ::description                            string?)
(s/def ::url                                    string?)
(s/def ::email                                  string?)
(s/def ::http-status-codes                      number?)

(s/def ::schema                                 any?)
(s/def ::externalDocs                           (s/keys :req-un [::url]
                                                        :opt-un [::description]))

(s/def ::maximum                                number?)
(s/def ::exclusiveMaximum                       boolean?)
(s/def ::minimum                                number?)
(s/def ::exclusiveMinimum                       boolean?)
(s/def ::maxLength                              integer?)
(s/def ::minLength                              integer?)
(s/def ::pattern                                string?)
(s/def ::maxItems                               integer?)
(s/def ::minItems                               integer?)
(s/def ::uniqueItems                            boolean?)
(s/def ::enum                                   (s/coll-of any?))
(s/def ::multipleOf                             number?)


(s/def ::data-type-transform (s/keys :req-un [:spectator.swagger.operation.parameter/type]
                                     :opt-un [:spectator.swagger.operation.parameter/format]))

(def default-data-type-transforms
  {'clojure.core/int?        {:type :integer :format :int64}
   'clojure.core/nat-int?    {:type :integer :format :int64}
   'clojure.core/float?      {:type :number :format :float}
   'clojure.core/double?     {:type :number :format :double}
   'clojure.core/bigdec?     {:type :number :format :double}
   'clojure.core/boolean?    {:type :boolean}
   'clojure.core/string?     {:type :string}
   'spectator.core/local-date? {:type :string :format :date}
   'spectator.core/future-local-date? {:type :string :format :date}
   'clojure.core/inst?       {:type :string :format :date-time}
   'spectator.core/string-1? {:type :string}})

(s/fdef get-all-keys-merge
        :args (s/cat :merge-spec-form :spectator.core/merge-form #_(s/or :keyword keyword? :mergeform :spectator.core/merge-form))
        :ret string?)

(defn get-all-keys-merge [merge-spec-form]
  "Compbine all keys (:req/:req-un/:opt/:opt-un) from a merge form in a single list"
  (->> merge-spec-form
       rest
       (map (fn [x] (s/conform :spectator.core/keys-form (s/form (if (keyword? x) x (eval x))))))
       (map #(-> % :options))
       (apply (partial merge-with concat))
       vals
       flatten
       )
  )

(defn get-req-keys-merge [merge-spec-form]
  "Combine all keys (:req/:req-un/:opt/:opt-un) from a merge form in a single list"
  (as-> merge-spec-form $
        (rest $)
        (map (fn [x] (s/conform :spectator.core/keys-form (s/form (if (keyword? x) x (eval x))))) $)
        (map #(-> % :options) $)
        (apply (partial merge-with concat) $)
        (select-keys $ [:req :req-un])
        (vals $)
        (flatten $)))

(s/fdef get-all-keys
        :args (s/cat :keys-spec-form :spectator.core/keys-form)
        :ret  (s/coll-of keyword?))

(defn get-all-keys [keys-spec-form]
  "Compbine all keys (:req/:req-un/:opt/:opt-un) from a keys form in a single list"
  (let [conformed (s/conform ::sp/keys-form keys-spec-form)]
    (if-not (= conformed :clojure.spec.alpha/invalid)
      (let [keys (:options conformed)]
        (concat (:req keys)
                (:req-un keys)
                (:opt keys)
                (:opt-un keys)))
      [])))

(defn get-keys [keys-spec-form key-types]
  "Compbine all keys from keys-types (i.e. :req/:req-un/:opt/:opt-un) in a single list"
  (let [conformed (s/conform ::sp/keys-form keys-spec-form)]
    (if-not (= conformed :clojure.spec.alpha/invalid)
      (-> conformed
          :options
          (select-keys key-types)
          vals
          flatten)
      [])))

(defn get-required [keys-spec-form]
  (let [conformed (s/conform ::sp/keys-form keys-spec-form)]
    (if-not (= conformed :clojure.spec.alpha/invalid)
      (let [keys (:options conformed)]
        (concat (:req keys)
                (:req-un keys)))
      [])))

(s/fdef spec->swagger-definition
        :args (s/cat :spec-name-or-form (s/or :keyword keyword?
                                              :seq seq?)
                     :data-type-transforms (s/? (s/map-of symbol? ::data-type-transform)))
        :ret  ::swagger-root)

(defn spec->swagger-definition
  ([spec-name-or-form]
   (s/form spec-name-or-form)
   (spec->swagger-definition spec-name-or-form default-data-type-transforms))
  ([spec-name-or-form data-type-transforms]
   (let [spec-name (when (keyword? spec-name-or-form)
                     spec-name-or-form)
         spec-form (if spec-name
                     (s/form spec-name)
                     (if (s/spec? spec-name-or-form)
                       (s/form spec-name-or-form)
                       spec-name-or-form))]
     (let [val (cond
                 (sp/set-spec? spec-form)   {:enum (into [] spec-form)}

                 (sp/merge-spec? spec-form) (merge {:type :object
                                                    :properties
                                                          (->> (get-all-keys-merge spec-form)
                                                               (map (fn [child-spec-names]
                                                                      (spec->swagger-definition child-spec-names data-type-transforms)))
                                                               (apply merge))}
                                                   (let [required-keys (get-req-keys-merge spec-form)]
                                                     (when (not (empty? required-keys))
                                                       {:required (mapv (fn [kw] (sp/remove-namespace kw)) required-keys)})))

                 (sp/keys-spec? spec-form) (merge {:type :object
                                                   :properties
                                                         (->> (get-all-keys spec-form)
                                                              (map (fn [child-spec-names]
                                                                     (spec->swagger-definition child-spec-names data-type-transforms)))
                                                              (apply merge))}
                                                  (let [required-keys (get-required spec-form)]
                                                    (when (not (empty? required-keys))
                                                      {:required (mapv (fn [kw] (sp/remove-namespace kw)) required-keys)})))

                 (sp/coll-spec? spec-form) (let [conformed (s/conform :spectator.core/every-form spec-form)
                                                 min-count (-> conformed :options :min-count)]
                                             (merge {:type  :array
                                                     :items (spec->swagger-definition (second spec-form) data-type-transforms)}
                                                    (when-not (nil? min-count)
                                                      {:minItems min-count})))

                 ;(sp/and-spec? spec-form) (expand-swagger (second spec-form) data-type-transforms)
                 :else (let [transformed (get data-type-transforms spec-form)]
                         (merge {:type (or (:type transformed) spec-form)}
                                (when (:format transformed)
                                  {:format (:format transformed)}))))]
       (if spec-name
         {(sp/remove-namespace spec-name) val}
         val)))))

#_(defn get-spec-names
  [swagger-root]
  (->> swagger-root
       (specter/select
         (specter/walker #(and
                           (map? %)
                           (contains? % :spec))))
       (map #(:spec %))
       set))

(defn spec->$ref
  [form]
  "If form is a map, containing a key named :spec, change it to a Swagger $ref object else return the original form"
  (if (and (map? form)
           (contains? form :spec))
    (-> form
        (dissoc :spec)
        (assoc :$ref (str "#/definitions/"
                          (-> (:spec form)
                              (clojure.string/split #"/")
                              last))))
    form))

(defn replace-spec-entries
  [swagger-root]
  (clojure.walk/postwalk
    spec->$ref
    swagger-root))

(defn add-definitions
  [swagger-root spec-names]
  )

(s/fdef swagger->json
        :args (s/cat :swagger ::swagger-root)
        :ret  string?)

;(defn swagger->json
;  [swagger]
;  (cheshire/generate-string swagger))
;
;(defn swagger->pretty-json
;  [swagger]
;  (cheshire/generate-string swagger {:pretty true}))

(s/fdef swagger->yaml
        :args (s/cat :swagger ::swagger-root)
        :ret  string?)

(defn swagger->yaml
  [swagger]
  (yaml/generate-string swagger :dumper-options {:flow-style :block}))

(defn generate
  []
  (gen/generate (s/gen ::swagger-root)))