(ns skalar.pool
  (:require [failjure.core :as f]
            [me.raynes.conch.low-level :as sh]))

(declare create-session close-session)

(def temp-dir-attrs
  (into-array java.nio.file.attribute.FileAttribute []))

(defprotocol Poolable
  (reserve-session [this] "reserves session from pool adding one if capacity has been reached")
  (release-session [this session] "releases session potentially removing it from pool"))

(defrecord Pool [base capacity max sessions]
  Poolable

  (reserve-session [this]
    (let [session  (apply min-key (comp deref :attached) sessions)
          created  (count sessions)
          attached (:attached session)]

      (if (or (= created max)
              (< @attached capacity))
        (do (swap! attached inc) session)
        (first
         (swap! sessions conj (create-session (inc created) 1))))))

  (release-session [this session]
    (let [attached (swap! (:attached session) dec)]
      (if (or (< attached 0)
              (and (= attached 0)
                   (> (count sessions) base)))
        (let [id (:id (close-session session))]
          (swap! sessions (filter #(not (= (:id %1) id)) sessions)))
        sessions))))

(defn create-proc [tmp-dir]
  (let [proc (sh/proc "gm" "batch" "-feedback" "on" :dir tmp-dir)]
    (sh/read-line proc :out)
    proc))

(defn create-pool [base capacity max]
  (let [iter (take base (iterate inc 1))
        sessions (doall (map #(create-session % 0) iter))]
    (->Pool base capacity max sessions)))

(defn shutdown-pool [pool]
  (doseq [session @(:sessions pool)]
    (.release-session pool session)))

(defn close-session [session]
  (when-let [proc @(:process session)]
    (sh/done proc)
    (sh/destroy proc)
    session))

(defn create-session [n attached]
  (let [temp-dir (java.nio.file.Files/createTempDirectory "skalar-" temp-dir-attrs)
        temp-str (.toString temp-dir)]

    ;; delete temporary dir on jvm exit
    (.deleteOnExit (.toFile temp-dir))

    {:id       (str "gm-" n)
     :attached (atom attached)
     :process  (agent (create-proc temp-str))
     :temp-dir temp-str}))

(defn send-cmd [pool cmd {:keys [output size]}]
  (let [session  (reserve-session pool)
        process  (:process session)
        temp-dir (:temp-dir session)
        result   (promise)]

    (send-off process
              (fn [proc args]
                (deliver result
                         (f/try*

                          ;; process a command
                          (sh/feed-from-string proc args)

                          ;; gather a feedback from session
                          (let [feedback (sh/read-line proc :out)]
                            (release-session pool session)

                            (if (= "FAIL" feedback)
                              (f/fail "Error while processing. Possibly not a valid image.")
                              {:size size
                               :path (->> (into-array String [size output])
                                          (java.nio.file.Paths/get temp-dir)
                                          (.toAbsolutePath)
                                          (.toString))}))))
                proc)
              cmd)
    result))
