(ns degree9.kubernetes
  (:refer-clojure :exclude [namespace])
  (:require
    [cljs.nodejs :as node]
    [goog.object :as obj]
    [feathers.errors :as error]))

;; Kubernetes API ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(def k8s (node/require "@kubernetes/client-node"))

(def fs (node/require "fs"))

(def path (node/require "path"))

(def k8s-svc (str "/var/run/secrets/kubernetes.io/serviceaccount"))

(def k8s-crt (str k8s-svc "/ca.crt"))

(def k8s-token (str k8s-svc "/token"))

(def k8s-host js/process.env.KUBERNETES_SERVICE_HOST)

(def k8s-port js/process.env.KUBERNETES_SERVICE_PORT)

(def k8s-config (or js/process.env.KUBECONFIG
                    (.join path js/process.env.HOME ".kube" "config")))

(defn- exists [path]
  (.existsSync fs path))

(defn- read-file [path]
  (.readFileSync fs path))

(defn kube-config []
  (let [config (k8s.KubeConfig.)]
    (when (exists k8s-config)
      (.loadFromFile config k8s-config))
    config))

(def CoreV1Api k8s.Core_v1Api)

(defn cluster-config [api crt token]
  (.log js/console api crt token))

(defn file-config [api]
  (let [config  (kube-config)
        cluster (.getCurrentCluster config)
        server  (obj/get cluster "server")
        k8s-api (api. server)]
    (.setDefaultAuthentication k8s-api config)
    k8s-api))

(defn config [api]
  (cond
    (and k8s-host k8s-port)   (cluster-config api k8s-host k8s-port)
    (exists k8s-config) (file-config api k8s-config)
    :else (api "http://localhost:8080")))

(config CoreV1Api)

(defn watch-error [err]
  (when err
    (.error js/console err)))

(defn watcher
  ([path callback])
  ([path callback error-callback])
  ([path opts callback error-callback]
   (.watch (k8s.Watch. (kube-config))
     path
     (clj->js opts)
     callback
     error-callback)))


(.watch (k8s.Watch. (kube-config))
  "/apis/kate.degree9.io/v1/tenants/"
  #js{}
  #(.log js/console %1 %2)
  watch-error)

(.log js/console (kube-config k8s.Core_v1Api))

;(defn get-k8sconfig
;  (let [kenv js/process.env.KUBECONFIG
;        kconfig (.join path js/process.env.HOME ".kube" "config")
;    (cond
;      (exists kenv) kenv
;      (exists kconfig) kconfig))

;(defn kubernetes [& [api config]]
;  (if-let [svc-token (and (exists svc-token) svc-token)]
      ;(exists kconfig)
      ;(let [kconf   (config-file kconfig)
      ;      kserver (goog.object/get kconf "server")
      ;      k8s-api (k8s.Core_v1Api. kserver)
      ;  (.setDefaultAuthentication k8s-api kconf)
      ;  k8s-api
;      (exists (:token svc-account)) (.log js/console "K8S Api Should Load service account token."))))



;(.log js/console "K8S" (kubernetes))


(def Config k8s.Config)

(def apps-api (k8s.Apps_v1Api.))

(def core-api (k8s.Config.defaultClient))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Kubernetes Helpers ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn- k8s->clj
  "Converts Kubernetes response to ClojureScript."
  [k8s]
  (js->clj k8s :keywordize-keys true))

(defn- k8s-response [res]
  (clj->js (:body (k8s->clj res))))

(defn- k8s-error
  "Format a Kubernetes error as a [message details] pair."
  [err]
  (let [{:keys [body]} (k8s->clj err)
        message (:message body)
        details (:details body)]
    [message (clj->js details)]))

(defn- not-found
  "Emits a NotFound error from a Kubernetes error response."
  [err]
  (let [[message details] (k8s-error err)]
    (error/not-found message details)))

(defn- already-exists
  "Emits a Conflict error from a Kubernetes error response."
  [err]
  (let [[message details] (k8s-error err)]
    (error/conflict message details)))

(defn- create-namespace
  "Create a Kubernetes namespace."
  [data]
  (-> core-api
    (.createNamespace data)
    (.then k8s-response)
    (.catch already-exists)))

(defn- read-namespace
  "Read a Kubernetes namespace."
  [name]
  (-> core-api
    (.readNamespace name)
    (.then k8s-response)
    (.catch not-found)))

(defn- create-deployment
  "Create a Kubernetes deployment."
  [data namespace]
  (-> core-api
    (.createNamespacedDeployment data namespace)
    (.then k8s-response)
    (.catch already-exists)))

(defn- read-deployment
  "List Deployments from a Kubernetes namespace."
  [name namespace]
  (-> apps-api
    (.readNamespacedDeployment name namespace)
    (.then k8s-response)
    (.catch prn)))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Kubernetes Services ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn namespace [& [opts]]
  (let []
    (reify
      Object
      ;(find [this params]
      ;  ())
      (get [this id & [params]]
        (read-namespace id))
      (create [this data & [params]]
        (create-namespace data)))))
      ;(remove [this id params]
      ;  ()))))

(defn deployment [& [opts]]
  (let []
    (reify
      Object
      ;(find [this params]
      ;  ())
      (get [this id & [{:keys [] :as params}]]
        (.log js/console id params)
        (read-deployment id nil)))))
      ;(create [this data & [params]]
      ;  (create-namespace data))))
      ;(remove [this id params]
      ;  ()))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
