(ns degree9.env
  "Load configuration variables from process.env or .env file."
  (:refer-clojure :exclude [get keys])
  (:require [cuerdas.core :as str]
            [clojure.string :as cstr]
            [goog.object :as obj]
            [degree9.debug :as dbg]
            ["dotenv" :as dotenv]
            ["fs" :as fs]
            ["path" :as path]))

(dbg/defdebug debug "degree9:enterprise:env")

;; Env State ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(def ^:dynamic *env* (atom nil))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- read-file [path]
  (.readFileSync fs path #js{:encoding "utf8"}))

(defn- split-kv [kvstr]
  (cstr/split kvstr #"=" 1))

(defn- split-config [config]
  (->> (cstr/split-lines config)
       (map split-kv)
       (into {})))

(defn- dot-env! []
  (split-config (read-file ".env")))

(defn- node-env! []
  (js->clj js/process.env))

(defn init! []
  (reset! *env* (merge (dot-env!) (node-env!))))

(defn keys
  "Return all keys from the process.env object."
  []
  (->> (env-obj)
    (js-keys)
    (js->clj)
    (map str/kebab)
    (map keyword)))

(defn get
  "Return the environment value for `key` or `default`."
  ([key] (get key nil))
  ([key default] (obj/get @*env* (-> key name str/snake str/upper) default)))
