(ns degree9.es6
  (:refer-clojure :exclude [class])
  (:require [cljs.analyzer :as ana]
            [cljs.compiler :as compiler]))

;; Native ES6 Class compiler extension ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(alter-var-root #'ana/specials #(conj % 'class* 'class-method* 'class-ctor*))

(defmethod ana/parse 'class*
  [op env [_ sym & exprs :as form] _ _]
  {:env env
   :op :class
   :children [:methods]
   :methods (ana/disallowing-recur
                 (->> (rest exprs)
                      (mapv #(ana/analyze (assoc env :context :method) %))))
   :form form
   :class sym})

(defmethod compiler/emit* :class
  [{:keys [methods class]}]
  (compiler/emits "class ")
  (compiler/emits class)
  (compiler/emits " {")
  (doseq [m methods]
    (compiler/emitln m))
  (compiler/emits "}"))

(defmethod ana/parse 'class-ctor*
  [op env [_ _ args & exprs :as form] _ _]
  {:env env
   :op :class-ctor
   :children [:statements]
   :statements (ana/disallowing-recur
                 (->> (rest exprs)
                      (mapv #(ana/analyze (assoc env :context :statement) %))))
   :form form
   :args args})

(defmethod ana/parse 'class-method*
  [op env [_ method & exprs :as form] _ _]
  {:env env
   :op :method
   :children [:statements]
   :statements (ana/disallowing-recur
                 (->> (rest exprs)
                      (mapv #(ana/analyze (assoc env :context :statement) %))))
   :form form
   :method method})

(defmethod compiler/emit* :method
  [{:keys [statements method]}]
  (compiler/emits "class ")
  (compiler/emits method)
  (compiler/emits " {")
  (doseq [s statements]
    (compiler/emitln s))
  (compiler/emits "}"))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Public Async/Await API ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmacro class
  "Wraps body in native es6 class block."
  [name & body]
  `(let [~'constructor class-ctor*
         ~'method class-method*]
     (~'class* ~name ~@body)))

(defmacro defclass
  "Define a native es6 class."
  [name & body]
  `(def ~name (class ~name ~@body)))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
