(ns degree9.es6
  (:refer-clojure :exclude [class])
  (:require [cljs.analyzer :as ana]
            [cljs.compiler :as compiler]))

;; Native ES6 Class compiler extension ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(alter-var-root #'ana/specials #(conj % 'class* 'method* 'super*))

(defmethod ana/parse 'class*
  [op env [_ class extends methods :as form] _ _]
  (let [methods (map (fn [[method params & body]] {:method method :params params :body body}) methods)]
    (prn methods)
    {:env env
     :op :class
     :children [:methods]
     :methods (ana/disallowing-recur
                 (->> (rest methods)
                      (mapv #(ana/analyze (assoc env :context :method) %))))
     :ctor (ana/disallowing-recur
                 (->> (first methods)
                      (mapv #(ana/analyze (assoc env :context :constructor) %))))
     :extends extends
     :form form
     :class class}))

(defmethod compiler/emit* :class
  [{:keys [ctor methods class extends]}]
  (let [extends (when extends (str " extends " extends))]
    (compiler/emitln "class " class extends " {")
    (compiler/emitln "constructor( " (interpose "," (map compiler/munge args)) ") {")
    (doseq [c ctor]
      (compiler/emitln c))
    (compiler/emits "}")
    ; (doseq [m methods]
    ;   (compiler/emitln m))
    (compiler/emits "}")))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ES6 Class API ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmacro class
  "Create a named or unnamed javascript class. (es6+)"
  ([] (class constructor-fn {}))
  ([methods] (class nil methods))
  ([name methods] (class name nil methods))
  ([name extends methods] `(~'class* ~name ~extends ~methods)))

(defmacro defclass
  "Define a named javascript class. (es6+)"
  ([name & [extends & methods :as body]]
   (let [extends (when (symbol? extends) extends)
         methods (if extends methods body)]
     `(def ~name (~'class ~name ~extends ~methods)))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
