(ns degree9.es6
  (:refer-clojure :exclude [class])
  (:require [cljs.analyzer :as ana]
            [cljs.compiler :as compiler]))

;; Native ES6 Class compiler extension ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(alter-var-root #'ana/specials #(conj % 'class* 'method* 'super*))

(defmethod ana/parse 'method*
  [op env [_ method params & exprs :as form] _ _]
  {:env env
   :op :method
   :children [:statements]
   :form form
   :method method
   :params params
   :statements (ana/disallowing-recur
                 (->> exprs
                      (mapv #(ana/analyze (assoc env :context :statement) %))))})

(defmethod compiler/emit* :method
  [{:keys [method params statements]}]
  (compiler/emitln method "(" (interpose "," (map compiler/munge params)) "){")
  (doseq [s statements]
    (compiler/emitln s))
  (compiler/emits "}"))

(defmethod ana/parse 'class*
  [op env [_ class extends methods :as form] _ _]
  {:env env
   :op :class
   :children [:methods]
   :form form
   :class class
   :extends extends
   :methods (ana/disallowing-recur
               (->> methods
                    (mapv #(conj % degree9.es6/method))
                    (mapv #(ana/analyze (assoc env :context :method) %))))})

(defmethod compiler/emit* :class
  [{:keys [class extends methods]}]
  (let [extends (when extends (str " extends " extends))]
    (compiler/emitln "class " class extends " {")
    (doseq [m methods]
      (compiler/emitln m))
    (compiler/emits "}")))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ES6 Class API ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmacro method
  "Create a javascript class method. (es6+)

   Javascript DOES NOT support overloading methods, since we are creating native
   method blocks, we also do not support overloading methods."
  ([] (class constructor-fn {}))
  ([methods] (class nil methods))
  ([name methods] (class name nil methods))
  ([name extends methods] `(~'class* ~name ~extends ~methods)))

(defmacro class
  "Create a named or unnamed javascript class. (es6+)"
  ([] (class constructor-fn {}))
  ([methods] (class nil methods))
  ([name methods] (class name nil methods))
  ([name extends methods] `(~'class* ~name ~extends ~methods)))

(defmacro defclass
  "Define a named javascript class. (es6+)"
  ([name & [extends & methods :as body]]
   (let [extends (when (symbol? extends) extends)
         methods (if extends methods body)]
     `(def ~name (~'class ~name ~extends ~methods)))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
