(ns degree9.es6
  (:refer-clojure :exclude [class])
  (:require [cljs.analyzer :as analyzer]
            [cljs.compiler :as compiler]))

;; Extends Analyzer Special Forms ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(alter-var-root #'analyzer/specials #(conj % 'class* 'super*))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Native ES6 Class Super compiler extension ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmethod analyzer/parse 'super*
  [op env [_  & params :as form] _ _]
  {:env env
   :op :super
   :form form
   :params (analyzer/disallowing-recur
            (mapv #(analyzer/analyze env %) params))})

(defmethod compiler/emit* :super
  [{:keys [params env]}]
  (compiler/emits "super(" (interpose "," params) ")"))

(defmacro super
  "Call the parent class from a new extended class method. (es6+)"
  [& params]
  `(~'super* ~@params))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Native ES6 Class compiler extension ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmethod analyzer/parse 'method*
  [op env [_ params & body :as form] _ _]
  {:env env
   :op :class-method
   :children [:exprs]
   :form form
   :params params
   :exprs (analyzer/disallowing-recur
            (mapv (fn [ctor] (analyzer/analyze (reduce #(assoc-in %1 [:locals %2] {:name %2}) env params) ctor)) ctor))})

(defmethod analyzer/parse 'class*
  [op env [_ class extends & methods :as form] _ _]
  (analyzer/disallowing-recur
    {:env env
     :op :class
     :children [:ctor]
     :form form
     :class class
     :extends (when extends (analyzer/analyze-symbol env extends))
     :methods methods}))

(defmethod compiler/emit* :class
  [{:keys [class extends params ctor]}]
  (compiler/emits "class " class)
  (when extends (compiler/emits " extends " extends))
  (compiler/emitln " {")
  (compiler/emitln "constructor(" (interpose "," params) "){")
  (doseq [c ctor]
    (compiler/emitln c))
  (compiler/emitln "}")
  (compiler/emits "}"))

(defmacro ^:private class
  "Create a named or unnamed javascript class. (es6+)"
  ([ctor] (class nil ctor))
  ([name ctor] (class name nil ctor))
  ([name extends & methods]
   `(~'class* ~name ~extends ~@(map method* methods))))

(defmacro defclass
  "Define a named javascript class. (es6+)"
  ([name ctor] (defclass name nil ctor))
  ([name extends ctor]
   `(def ~name (class ~name ~extends ~ctor))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
