(ns degree9.es6
  (:refer-clojure :exclude [class])
  (:require [cljs.analyzer :as analyzer]
            [cljs.compiler :as compiler]))

;; Extends Analyzer Special Forms ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(alter-var-root #'analyzer/specials #(conj % 'js-class* 'js-method* 'js-super*))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Native ES6 Class Super ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmacro ^:private js-super [& args]
  `(js* "super"))

(defmacro super-as
  "Defines a scope where JavaScript's implicit \"super\" is bound to the name provided."
  [name & body]
  `(let [~name (js-super)]
     ~@body))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Native ES6 Class compiler extension ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn ^:private assoc-locals [env local]
  (assoc-in env [:locals local] {:name local}))

(defn ^:private analyse-with-locals [env locals exp]
  (analyzer/analyze (reduce assoc-locals env locals) exp))

(defmethod analyzer/parse 'js-method*
  [op env [_ method params & exprs :as form] _ _]
  (analyzer/disallowing-recur
    {:env env
     :op :js-method
     :children [:exprs]
     :form form
     :method method
     :params params
     :exprs (mapv (partial analyse-with-locals (assoc env :context :expr) params) exprs)}))

(defmethod compiler/emit* :js-method
  [{:keys [method params exprs]}]
  (compiler/emitln method "(" (interpose "," params) "){")
  (doseq [e exprs]
    (compiler/emitln e))
  (compiler/emitln "}"))

(defn ^:private methodize [[method params & body]]
  `(~'js-method* ~method ~params ~@body))

(defmethod analyzer/parse 'js-class*
  [op env [_ class ctor & methods :as form] _ _]
  (prn ctor)
  (analyzer/disallowing-recur
    {:env env
     :op :js-class
     :children [:methods]
     :form form
     :class class
     :ctor ctor
     :methods (mapv (partial analyzer/analyze env) methods)}))

(defmethod compiler/emit* :js-class
  [{:keys [env class ctor methods]}]
  (when (= :return (:context env))
    (compiler/emits "return "))
  (compiler/emitln "class " class " {")
  (doseq [m methods]
    (compiler/emitln m))
  (compiler/emits "}"))

(defmacro ^:private class
  "Create a named or unnamed javascript class. (es6+)"
  [& [name & [extends :as methods] :as options]]
  (let [name    (when (symbol? name) name)
        extends (when (symbol? extends) extends)
        methods (if name methods options)
        methods (if extends (rest methods) methods)
        ctor    (-> (map (juxt first rest) methods)
                    (get 'constructor))]
   `(~'js-class* ~name ~ctor ~@(map methodize methods))))

(defmacro defclass
  "Define a named javascript class. (es6+)"
  ([name ctor] (defclass name nil ctor))
  ([name extends ctor]
   `(def ~name (class ~name ~extends ~ctor))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
