(ns degree9.routing
  (:require [goog.events :as events]
            [goog.dom :as dom]
            [javelin.core :as j]
            [degree9.object :as obj]
            [degree9.string :as str])
  (:require-macros degree9.routing))

;; Window History Object ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(def ^:dynamic *history* (.-history js/window))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; HTML5 History State ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn- state-cell []
  (let [history (j/cell (.-state *history*))]
    (j/with-let [history= (j/cell= history (fn [state] (reset! history state)))]
      (events/listen js/window events/EventType.POPSTATE
        (fn [event] (reset! history= (.-state event)))))))

(def state (state-cell))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; URL ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn- url-cell []
  (let [url (j/cell (js/URL. (.. js/window -location)))]
    (j/with-let [url= (j/cell= url (fn [loc] (reset! url (js/URL. loc))))]
      (events/listen js/window events/EventType.POPSTATE
        (fn [event] (reset! url= (.. js/window -location)))))))

(def url (url-cell))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; URLSearchParams ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn- search->clj [search]
  (let [entries (.entries search)]
    (js->clj (obj/from-entries entries))))

(defn- clj->search [data]
  (js/URLSearchParams. (clj->js data)))

(defn- search-cell []
  (let [params (j/cell (js/URLSearchParams. (.. js/window -location -search)))]
    (j/with-let [params= (j/cell= params (fn [search] (reset! params (js/URLSearchParams. search))))]
      (events/listen js/window events/EventType.POPSTATE
        (fn [event] (reset! params= (.. js/window -location -search)))))))

(def search (search-cell))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; URL Pathname ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn- path-cell []
  (let [url (js/URL. (.. js/window -location))
        path (j/cell (.-pathname url))]
    (j/with-let [path= (j/cell= path (fn [loc] (reset! path (.-pathname loc))))]
      (events/listen js/window events/EventType.POPSTATE
        (fn [event] (reset! path= (.. js/window -location)))))))

(def path (path-cell))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; HTML5 History Helpers ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn push! [state title path]
  (.pushState *history* state title path))

(defn replace! [state title path]
  (.replaceState *history* state title path))

(defn popstate! [state]
  (.dispatchEvent js/window
    (js/PopStateEvent. "popstate" (clj->js {:state state}))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Routing Public API ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn route!
  "Provides client (SPA) routing without reloading the page. "
  ([path] (route! path nil))
  ([path query] (route! path query @state))
  ([path query state]
   (let [title  (.. js/document -title)
         state  (clj->js state)
         search (clj->search query)
         token  (if query (str/join "?" [path (str search)]) path)]
     (j/dosync
       (push! state title token)
       (popstate! state)))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
