(ns degree9.browser.indexed-db
  (:require [javelin.core :as j]
            [degree9.object :as obj]
            [degree9.browser :as bom]
            [degree9.browser.indexed-db.factory :as factory]
            [degree9.browser.indexed-db.opendbrequest :as oreq]
            [degree9.browser.indexed-db.objectstore :as ostore]
            [degree9.browser.indexed-db.database :as db]
            [degree9.browser.indexed-db.transaction :as tx])
  (:require-macros hoplon.indexed-db))

(defn open-database
  ([name] (factory/open (bom/indexed-db) name))
  ([name version] (factory/open (bom/indexed-db) name version)))

(defn create-stores [db stores]
  (let [current (db/objectStoreNames db)]
    (doseq [store stores]
      (when-not (.contains current store)
        (db/createObjectStore db store)))))

(defn delete-stores [db stores]
  (let [current (db/objectStoreNames db)]
    (doseq [store current]
      (when-not (contains? (set stores) store)
        (db/deleteObjectStore db store)))))

(defn get-result [event]
  (:result (:target event)))

(defn get-version [event]
  [(:oldVersion event) (:newVersion event)])

(defn when-success [tx callback]
  (tx/onsuccess tx callback))

(defn when-error [tx callback]
  (tx/onerror tx callback))

(defn when-upgrading [tx callback]
  (tx/onupgradeneeded tx callback))

(defn object-store= [req store key]
  (let [idb (j/cell nil)
        ostore (j/cell nil)
        ostore! (partial reset! ostore)]
    (when-success req
      (fn [event]
        (reset! idb (get-result event))
        (let [store (-> @idb (db/transaction store) (tx/objectStore store))]
          (when-success (ostore/get store key)
            (fn [event]
              (let [result (get-result event)]
                (ostore! (js->clj result))))))))
    (j/cell= ostore
      (fn [val]
        (let [store (-> @idb (db/transaction store "readwrite") (tx/objectStore store))]
          (when-success (ostore/put store (clj->js val) key)
            (fn [event]
              (ostore! val))))))))
