(ns degree9.service
  "A microservice proxy via websockets."
  (:require [degree9.socket-io :as io]
            ["@feathersjs/feathers" :as feathers]
            ["@feathersjs/socketio-client" :as socketio]
            [goog.object :as obj]
            [meta.server :as server]
            [degree9.debug :as dbg]))

(dbg/defdebug debug "degree9:enterprise:service")

(defn- log-proxy [method service & [params]]
  (debug "Proxy %s to remote service %s with params %s" method service params))

(defn- with-websockets [client socket]
  (doto client
    (.configure
      (socketio socket))))

(defn socket
  "Creates a socket.io client."
  ([uri] (io/io uri))
  ([uri opts] (io/io uri opts)))

(defn proxy
  "Creates a feathers websocket client."
  [socket]
  (with-websockets (feathers) socket))

(defn µservice [client path]
  (let [svc (.service client path)]
    (reify
      Object
      (setup [this app path]
        (debug "Setup proxy to remote service %s" path)
        (obj/set this :io svc))
      (find [this params]
        (log-proxy "find" path params)
        (.find (obj/get this :io) params))
      (get [this id params]
        (log-proxy "get" path params)
        (.get (obj/get this :io) id params))
      (create [this data params]
        (log-proxy "create" path params)
        (.create (obj/get this :io) data params))
      (update [this id data params]
        (log-proxy "update" path params)
        (.update (obj/get this :io) id data params))
      (patch [this id data params]
        (log-proxy "patch" path params)
        (.patch (obj/get this :io) id data params))
      (remove [this id params]
        (log-proxy "remove" path params)
        (.remove (obj/get this :io) id params)))))

(defn api
  "Mount a remote Microservice to a local endpoint."
  [app path proxy service hooks]
  (debug "Mount remote microservice at %s" path)
  (server/api app path (µservice proxy service) hooks))

;; Multi-Service ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn- api-service [app [prefix url services] hooks]
  (reduce (fn [app service] (api app (str prefix service) url service hooks)) app services))

(defn- reduce-apis [app services hooks]
  (reduce (fn [app service] (api-service app service hooks)) app services))

(defn multi-service
  "Mount multiple remote services from different servers."
  [app specs & [hooks]]
  (reduce-apis app specs hooks))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
