(ns firebase-cljs.auth
  (:refer-clojure :exclude [name])
  (:require [cljsjs.firebase]))

(defprotocol FirebaseAuth

  (get-app
    [_]
    "The App associated with the Auth service instance.")

  (current-user
    [_]
    "The currently signed-in user (or null).")

  (apply-actioncode
    [_ code]
    "Applies a verification code sent to the user by email or other out-of-band mechanism.")

  (check-actioncode
    [_ code]
    "Checks a verification code sent to the user by email or other out-of-band mechanism.")

  (confirm-pass-reset
    [_ code pass]
    "Completes the password reset process, given a confirmation code and new password.")

  (create-user
    [_ email pass]
    "Creates a new user account associated with the specified email address and password.")

  (providers-by-email
    [_ email]
    "Gets the list of provider IDs that can be used to sign in for the given email address. Useful for an 'identifier-first' sign-in flow.")

  (redirect-result
    [_]
    "Returns a UserCredential from the redirect-based sign-in flow.
    If sign-in succeeded, returns the signed in user. If sign-in was unsuccessful, fails with an error. If no redirect operation was called, returns a UserCredential with a null User.")

  (auth-changed
    [_ observer]
    [_ observer failure]
    [_ observer failure complete]
    "Adds an observer for auth state changes.")

  (send-pass-reset
    [_ email]
    "Sends a password reset email to the given email address.")

  (login-anon
    [_]
    "Asynchronously signs in as an anonymous user.
    If there is already an anonymous user signed in, that user will be returned; otherwise, a new anonymous user identity will be created and returned.")

  (login-cred
    [_ cred]
    "Asynchronously signs in with the given credentials.")

  (login-token
    [_ token]
    "Asynchronously signs in using a custom token.
    Custom tokens are used to integrate Firebase Auth with existing auth systems, and must be generated by the auth backend.")

  (login-userpass
    [_ email pass]
    "Asynchronously signs in using an email and password.")

  (login-popup
    [_ provider]
    "Authenticates a Firebase client using a popup-based OAuth authentication flow.")

  (login-redirect
    [_ provider]
    "Authenticates a Firebase client using a full-page redirect flow.")

  (logout
    [_]
    "Signs out the current user.")

  (verify-pass-reset
    [_ code]
    "Checks a password reset code sent to the user by email or other out-of-band mechanism."))

;(defprotocol FirebaseAuthCredential

;  (get-provider
;    [_]
;    "The authentication provider ID for the credential. For example, 'facebook.com', or 'google.com'."))

;(defprotocol FirebaseAuthProvider

;  (get-provider
;    [_]
;    "The authentication provider ID for the provider."))

(extend-type firebase.auth.Auth

  FirebaseAuth
  (get-app [auth] (.. auth -app))

  (current-user [auth] (.. auth -currentUser))

  (apply-actioncode [auth code] (.. auth (applyActionCode code)))

  (check-actioncode [auth code] (.. auth (checkActionCode code)))

  (confirm-pass-reset [auth code pass] (.. auth (confirmPasswordReset code pass)))

  (create-user [auth email pass] (.. auth (createUserWithEmailAndPassword email pass)))

  (providers-by-email [auth email] (.. auth (fetchProvidersForEmail email)))

  (redirect-result [auth] (.. auth getRedirectResult))

  (auth-changed
    ([auth observer]
     (.. auth (onAuthStateChanged observer)))
    ([auth observer failure]
     (.. auth (onAuthStateChanged observer failure)))
    ([auth observer failure complete]
     (.. auth (onAuthStateChanged observer failure complete))))

  (send-pass-reset [auth email] (.. auth (sendPasswordResetEmail email)))

  (login-anon [auth] (.. auth signInAnonymously))

  (login-cred [auth cred] (.. auth (signInWithCredential cred)))

  (login-token [auth token] (.. auth (signInWithCustomToken token)))

  (login-userpass [auth email pass] (.. auth (signInWithEmailAndPassword email pass)))

  (login-popup [auth provider] (.. auth (signInWithPopup provider)))

  (login-redirect [auth provider] (.. auth (signInWithRedirect provider)))

  (logout [auth] (.. auth signOut))

  (verify-pass-reset [auth code] (.. auth (verifyPasswordResetCode code))))

;(extend-type js/firebase.auth.AuthCredential

;  FirebaseAuthCredential
;  (get-provider [cred] (.. cred -provider)))

;(extend-type js/firebase.auth.AuthProvider

;  FirebaseAuthProvider
;  (get-provider [prov] (.. prov -providerId)))
