'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

var _feathersErrors = require('feathers-errors');

var _feathersErrors2 = _interopRequireDefault(_feathersErrors);

var _jwtDecode = require('jwt-decode');

var _jwtDecode2 = _interopRequireDefault(_jwtDecode);

var _debug = require('debug');

var _debug2 = _interopRequireDefault(_debug);

var _utils = require('./utils');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var debug = (0, _debug2.default)('feathers-authentication-client');

var Passport = function () {
  function Passport(app, options) {
    _classCallCheck(this, Passport);

    if (app.passport) {
      throw new Error('You have already registered authentication on this client app instance. You only need to do it once.');
    }

    Object.assign(this, {
      options: options,
      app: app,
      payloadIsValid: _utils.payloadIsValid,
      getCookie: _utils.getCookie,
      clearCookie: _utils.clearCookie,
      storage: app.get('storage') || this.getStorage(options.storage)
    });

    this.setJWT = this.setJWT.bind(this);

    app.set('storage', this.storage);
    this.getJWT().then(this.setJWT);

    this.setupSocketListeners();
  }

  _createClass(Passport, [{
    key: 'setupSocketListeners',
    value: function setupSocketListeners() {
      var _this = this;

      var app = this.app;
      var socket = app.io || app.primus;
      var emit = app.io ? 'emit' : 'send';
      var reconnected = app.io ? 'reconnect' : 'reconnected';

      if (!socket) {
        return;
      }

      socket.on(reconnected, function () {
        debug('Socket reconnected');

        // If socket was already authenticated then re-authenticate
        // it with the server automatically.
        if (socket.authenticated) {
          var data = {
            strategy: _this.options.jwtStrategy,
            accessToken: app.get('accessToken')
          };
          _this.authenticateSocket(data, socket, emit).then(_this.setJWT).catch(function (error) {
            debug('Error re-authenticating after socket reconnect', error);
            socket.authenticated = false;
            app.emit('reauthentication-error', error);
          });
        }
      });

      var socketUpgradeHandler = function socketUpgradeHandler() {
        socket.io.engine.on('upgrade', function () {
          debug('Socket upgrading');

          // If socket was already authenticated then re-authenticate
          // it with the server automatically.
          if (socket.authenticated) {
            var data = {
              strategy: _this.options.jwtStrategy,
              accessToken: app.get('accessToken')
            };

            _this.authenticateSocket(data, socket, emit).then(_this.setJWT).catch(function (error) {
              debug('Error re-authenticating after socket upgrade', error);
              socket.authenticated = false;
              app.emit('reauthentication-error', error);
            });
          }
        });
      };

      if (socket.io && socket.io.engine) {
        socketUpgradeHandler();
      } else {
        socket.on('connect', socketUpgradeHandler);
      }
    }
  }, {
    key: 'connected',
    value: function connected() {
      var _this2 = this;

      var app = this.app;

      if (app.rest) {
        return Promise.resolve();
      }

      var socket = app.io || app.primus;

      if (!socket) {
        return Promise.reject(new Error('It looks like your client connection has not been configured.'));
      }

      if (app.io && socket.connected || app.primus && socket.readyState === 3) {
        debug('Socket already connected');
        return Promise.resolve(socket);
      }

      return new Promise(function (resolve, reject) {
        var connected = app.primus ? 'open' : 'connect';
        var disconnect = app.io ? 'disconnect' : 'end';
        var timeout = setTimeout(function () {
          debug('Socket connection timed out');
          reject(new Error('Socket connection timed out'));
        }, _this2.options.timeout);

        debug('Waiting for socket connection');

        var handleDisconnect = function handleDisconnect() {
          debug('Socket disconnected before it could connect');
          socket.authenticated = false;
        };

        // If disconnect happens before `connect` the promise will be rejected.
        socket.once(disconnect, handleDisconnect);
        socket.once(connected, function () {
          debug('Socket connected');
          debug('Removing ' + disconnect + ' listener');
          socket.removeListener(disconnect, handleDisconnect);
          clearTimeout(timeout);
          resolve(socket);
        });
      });
    }
  }, {
    key: 'authenticate',
    value: function authenticate() {
      var _this3 = this;

      var credentials = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

      var app = this.app;
      var getCredentials = Promise.resolve(credentials);

      // If no strategy was given let's try to authenticate with a stored JWT
      if (!credentials.strategy) {
        if (credentials.accessToken) {
          credentials.strategy = this.options.jwtStrategy;
        } else {
          getCredentials = this.getJWT().then(function (accessToken) {
            if (!accessToken) {
              return Promise.reject(new _feathersErrors2.default.NotAuthenticated('Could not find stored JWT and no authentication strategy was given'));
            }
            return { strategy: _this3.options.jwtStrategy, accessToken: accessToken };
          });
        }
      }

      return getCredentials.then(function (credentials) {
        return _this3.connected(app).then(function (socket) {
          if (app.rest) {
            return app.service(_this3.options.path).create(credentials).then(_this3.setJWT);
          }

          var emit = app.io ? 'emit' : 'send';
          return _this3.authenticateSocket(credentials, socket, emit).then(_this3.setJWT);
        });
      }).then(function (payload) {
        app.emit('authenticated', payload);
        return payload;
      });
    }

    // Returns a promise that authenticates a socket

  }, {
    key: 'authenticateSocket',
    value: function authenticateSocket(credentials, socket, emit) {
      var _this4 = this;

      return new Promise(function (resolve, reject) {
        var timeout = setTimeout(function () {
          debug('authenticateSocket timed out');
          reject(new Error('Authentication timed out'));
        }, _this4.options.timeout);

        debug('Attempting to authenticate socket');
        socket[emit]('authenticate', credentials, function (error, data) {
          if (error) {
            return reject(error);
          }

          clearTimeout(timeout);
          socket.authenticated = true;
          debug('Socket authenticated!');

          resolve(data);
        });
      });
    }
  }, {
    key: 'logoutSocket',
    value: function logoutSocket(socket, emit) {
      var _this5 = this;

      return new Promise(function (resolve, reject) {
        var timeout = setTimeout(function () {
          debug('logoutSocket timed out');
          reject(new Error('Logout timed out'));
        }, _this5.options.timeout);

        socket[emit]('logout', function (error) {
          clearTimeout(timeout);
          socket.authenticated = false;

          if (error) {
            return reject(error);
          }

          resolve();
        });
      });
    }
  }, {
    key: 'logout',
    value: function logout() {
      var _this6 = this;

      var app = this.app;

      app.set('accessToken', null);
      this.clearCookie(this.options.cookie);

      // remove the accessToken from localStorage
      return Promise.resolve(app.get('storage').removeItem(this.options.storageKey)).then(function () {
        // If using sockets de-authenticate the socket
        if (app.io || app.primus) {
          var method = app.io ? 'emit' : 'send';
          var socket = app.io ? app.io : app.primus;

          return _this6.logoutSocket(socket, method);
        }
      }).then(function (result) {
        app.emit('logout', result);

        return result;
      });
    }
  }, {
    key: 'setJWT',
    value: function setJWT(data) {
      var accessToken = data && data.accessToken ? data.accessToken : data;

      if (accessToken) {
        this.app.set('accessToken', accessToken);
        this.app.get('storage').setItem(this.options.storageKey, accessToken);
      }

      return Promise.resolve(data);
    }
  }, {
    key: 'getJWT',
    value: function getJWT() {
      var _this7 = this;

      var app = this.app;
      return new Promise(function (resolve, reject) {
        var accessToken = app.get('accessToken');

        if (accessToken) {
          return resolve(accessToken);
        }

        return Promise.resolve(_this7.storage.getItem(_this7.options.storageKey)).then(function (jwt) {
          var token = jwt || _this7.getCookie(_this7.options.cookie);

          if (token && token !== 'null' && !_this7.payloadIsValid((0, _jwtDecode2.default)(token))) {
            token = undefined;
          }

          return resolve(token);
        }).catch(reject);
      });
    }

    // Pass a jwt token, get back a payload if it's valid.

  }, {
    key: 'verifyJWT',
    value: function verifyJWT(token) {
      if (typeof token !== 'string') {
        return Promise.reject(new Error('Token provided to verifyJWT is missing or not a string'));
      }

      try {
        var payload = (0, _jwtDecode2.default)(token);

        if (this.payloadIsValid(payload)) {
          return Promise.resolve(payload);
        }

        return Promise.reject(new Error('Invalid token: expired'));
      } catch (error) {
        return Promise.reject(new Error('Cannot decode malformed token.'));
      }
    }

    // Returns a storage implementation

  }, {
    key: 'getStorage',
    value: function getStorage(storage) {
      if (storage) {
        return storage;
      }

      return new _utils.Storage();
    }
  }]);

  return Passport;
}();

exports.default = Passport;
module.exports = exports['default'];