(ns kehaar.configure
  (:require [kehaar.rabbitmq :as rmq]
            [kehaar.wire-up :as wire-up]
            [clojure.core.async :as async]
            [langohr.basic :as lb]))

(def default-exchange "")
(def default-exchange-options {:auto-delete false :durable true})
(def default-prefetch-limit 1)
(def default-queue-options {:auto-delete false :durable true :exclusive false})
(def default-threshold 10)
(def default-timeout 1000)

(defn require-ns [sym]
  (let [ns' (symbol (namespace sym))]
    (when-not (find-ns ns')
      (require ns'))))

(defn configure!
  ""
  ([configuration]
   (let [connnection (-> configuration
                         :connection
                         rmq/connect-with-retries)]
     (configure! connnection (:kehaar configuration))))
  ([connection configuration]
   (let [{:keys [event-exchanges incoming-services external-services incoming-events outgoing-events]}
         configuration
         channels (atom [])]
     (doseq [{:keys [exchange type options]
              :or {type "topic"
                   options default-exchange-options}}
             event-exchanges]
       (let [rabbit-ch (wire-up/declare-events-exchange
                        connection
                        exchange
                        type
                        options)]
         (swap! channels conj rabbit-ch)))
     (doseq [{:keys [f threshold exchange queue queue-options threads
                     prefetch-limit response]
              :or {exchange default-exchange
                   response nil
                   prefetch-limit default-prefetch-limit
                   queue-options default-queue-options
                   threads wire-up/default-thread-count
                   threshold default-threshold}}
             incoming-services]
       (require-ns f)
       (let [in-chan (async/chan)
             out-chan (async/chan)
             f-var (find-var f)
             ignore-no-reply-to? (not response)
             streaming? (= response :streaming)
             rabbit-ch (wire-up/incoming-service connection
                                                 exchange
                                                 queue
                                                 queue-options
                                                 in-chan
                                                 out-chan
                                                 ignore-no-reply-to?)]
         (swap! channels conj rabbit-ch)
         (lb/qos rabbit-ch prefetch-limit)
         (if streaming?
           (wire-up/start-streaming-responder! connection
                                               in-chan
                                               out-chan
                                               f-var
                                               threshold
                                               threads)
           (wire-up/start-responder! in-chan
                                     out-chan
                                     f-var
                                     threads))))
     (doseq [{:keys [response exchange queue queue-options timeout f]
              :or {exchange default-exchange
                   queue-options default-queue-options
                   timeout default-timeout
                   response nil}}
             external-services]
       (require-ns f)
       (let [channel (async/chan 100)]
         (let [external-service-fn (cond
                                     (= response :streaming) wire-up/streaming-external-service
                                     (not response) wire-up/external-service-fire-and-forget
                                     :else wire-up/external-service)
               async->fn-fn (if (nil? response)
                              wire-up/async->fire-and-forget-fn
                              wire-up/async->fn)]
           (alter-var-root (find-var f)
                           (constantly (async->fn-fn channel)))
           (let [rabbit-ch (external-service-fn connection
                                                exchange
                                                queue
                                                queue-options
                                                timeout
                                                channel)]
             (swap! channels conj rabbit-ch)))))
     (doseq [{:keys [queue queue-options exchange routing-key timeout f
                     prefetch-limit]
              :or {exchange default-exchange
                   prefetch-limit default-prefetch-limit
                   queue-options default-queue-options
                   timeout default-timeout}}
             incoming-events]
       (require-ns f)
       (let [channel (async/chan 100)
             rabbit-ch (wire-up/incoming-events-channel connection
                                                        queue
                                                        queue-options
                                                        exchange
                                                        routing-key
                                                        channel
                                                        timeout)]
         (swap! channels conj rabbit-ch)
         (lb/qos rabbit-ch prefetch-limit)
         (wire-up/start-event-handler! channel (find-var f))))
     (doseq [{:keys [exchange routing-key channel]
              :or {exchange default-exchange}}
             outgoing-events]
       (require-ns channel)
       (let [rabbit-ch (wire-up/outgoing-events-channel
                        connection
                        exchange
                        routing-key
                        (var-get (find-var channel)))]
         (swap! channels conj rabbit-ch)))

     {:connections [connection]
      :channels @channels})))
