package dev.goodpay.ir.interfaces.issuer.v1.validator;

import dev.goodpay.ir.interfaces.common.ContractValidationResult;
import dev.goodpay.ir.interfaces.common.Country;
import dev.goodpay.ir.interfaces.common.ValidationError;
import dev.goodpay.ir.interfaces.common.validator.Validator;
import dev.goodpay.ir.interfaces.issuer.v1.IssuerV1;

import java.net.MalformedURLException;
import java.net.URL;

public class IssuerV1Validator implements Validator<IssuerV1> {

    public IssuerV1Validator() {
    }

    @Override
    public ContractValidationResult validateCreate(final IssuerV1 issuer) {
        final ContractValidationResult validationResult = new ContractValidationResult();
        identifierCheck(issuer, validationResult);
        countryCheck(issuer, validationResult);
        nameCheck(issuer, validationResult);
        urlCheck(issuer, validationResult);
        return validationResult;
    }

    @Override
    public ContractValidationResult validateUpdate(final IssuerV1 updateObject) {
        final ContractValidationResult result = new ContractValidationResult();
        if (updateObject.getUuid() == null) {
            result.registerError(ValidationError.missing("uuid"));
        }
        urlCheck(updateObject, result);
        return result;
    }

    private void urlCheck(final IssuerV1 issuer, final ContractValidationResult validationResult) {
        if (issuer.getUrl() != null) {
            try {
                final URL url = new URL(issuer.getUrl());
            } catch (final MalformedURLException e) {
                validationResult.registerError(ValidationError.invalid("url", "malformed URL"));
            }
        }
    }

    private void nameCheck(final IssuerV1 issuer, final ContractValidationResult validationResult) {
        if (issuer.getName() == null) {
            validationResult.registerError(ValidationError.missing("name"));
        }
    }

    private void countryCheck(final IssuerV1 issuer, final ContractValidationResult validationResult) {
        if (issuer.getCountry() == null) {
            validationResult.registerError(ValidationError.missing("country"));
        } else if (!Country.BY_ISO3_CODE.containsKey(issuer.getCountry().toLowerCase())) {
            validationResult.registerError(ValidationError.invalid("country", "Invalid iso3Code for country"));
        }
    }

    private void identifierCheck(final IssuerV1 issuer, final ContractValidationResult validationResult) {
        if (issuer.getIssuerIdentifier() == null) {
            validationResult.registerError(ValidationError.missing("identifier"));
        }
    }
}
