;; Copyright 2014-2017 Red Hat, Inc, and individual contributors.
;; SPDX-License-Identifier: Apache-2.0

(ns singular.util
  "Various utility functions."
  (:require [clojure.string         :as str]
            [clojure.java.io        :as io]
            [clojure.java.classpath :as cp]
            [dynapath.dynamic-classpath :as dp]
            [singular.internal.util :refer [try-resolve warn-deprecated *warn-on-deprecation*]]
            )
  (:import [org.projectodd.wunderboss
            DynamicClassLoader WunderBoss]))

(defn at-exit 
  "Registers `f` to be called when the application is either stopped
  or, if running within a container, undeployed. Used internally to
  shutdown various services, but can be used by application code as
  well."
[f]
  (WunderBoss/addShutdownAction f))

(defn options []
  (WunderBoss/options))

(defn service-registry []
  (get (options) "service-registry"))

;(try
  (let [base-url-classloader
               (assoc dp/base-readable-addable-classpath
                 :classpath-urls (fn [^DynamicClassLoader cl]
                                   (seq (.getURLs cl)))
                 :add-classpath-url (fn [^DynamicClassLoader cl url]
                                      (.addURL cl url)))]

           ;; if dynapath is available, make our classloader join the party
           (extend org.projectodd.wunderboss.DynamicClassLoader
             dp/DynamicClasspath
             base-url-classloader)

           ;; users of dynapath often search for the highest addable loader,
           ;; which in the container will be the AppClassLoader, and we really
           ;; want the DynamicClassLoader to be used instead. Anything added
           ;; to the AppClassLoader won't be seen, since JBoss Modules is
           ;; between the ACL and the app.
           (when (WunderBoss/inContainer)
             (extend sun.misc.Launcher$AppClassLoader
               dp/DynamicClasspath
               (assoc base-url-classloader
                 :can-add? (constantly false)))
             )
           )
;  (catch Exception _))

(defn reset
  "Resets the underlying WunderBoss layer.
   This stops and clears all services. Intended to be used from a repl or from tests."
  []
  (WunderBoss/shutdownAndReset))

(defn in-container?
  "Returns true if running inside a WildFly/EAP container."
  []
  (WunderBoss/inContainer))

(defn in-eap?
  "Returns true if running inside an EAP container."
  []
  (when-let [f (try-resolve 'singular.wildfly/in-eap?)]
    (f)))

(defn in-cluster?
  "Returns true if running inside a WildFly/EAP container that's part of a cluster"
  []
  (when-let [f (try-resolve 'singular.wildfly/in-cluster?)]
    (f)))

(defn reset-fixture
  "Invokes `f`, then calls [[reset]] if not [[in-container?]].

   Useful as a test fixture where you want to reset underlying state
   after a test run, but also run the same tests in-container (via fntest
   or other), where resetting state will disconnect the repl. In the
   in-container case, you rely on undeploy to reset the state."
  [f]
  (try
    (f)
    (finally
      (when-not (in-container?)
        (reset)
        ;; windows is slow to release closed ports, so we pause to allow that to happen
        (when (re-find #"(?i)^windows" (System/getProperty "os.name"))
          (Thread/sleep 100))))))

(defn app-name
  "Returns the name of the current application."
  []
  (get (WunderBoss/options) "deployment-name" ""))

(defn http-port
  "Returns the HTTP port for the embedded web server.

   Returns the correct port when in-container, and the :port value."
  ([]
     (http-port nil))
  ([options]
     (if-let [wf-port-fn (try-resolve 'singular.wildfly/http-port)]
       (wf-port-fn)
       (:port options))))

(defn context-path
  "Returns the over-arching context-path for the web server.

   Returns the servlet-context's context path in-container, and \"\"
   outside."
  []
  (if-let [path-fn (try-resolve 'singular.wildfly/context-path)]
    (path-fn)
    ""))

(defn messaging-remoting-port
  "Returns the port that HornetQ is listening on for remote connections.

   Returns the correct port when in-container, and the default (5445),
   outside."
  []
  (if-let [wf-port-fn (try-resolve 'singular.wildfly/messaging-remoting-port)]
    (wf-port-fn)
    (read-string (System/getProperty "hornetq.netty.port" "5445"))))

(defn classpath
  "Returns the effective classpath for the application."
  []
  (cp/classpath))

(defn dev-mode?
  "Returns true if the app is running in dev mode.

   This is controlled by the `LEIN_NO_DEV` environment variable."
  []
  (not (System/getenv "LEIN_NO_DEV")))

(defn set-bean-property
  "Calls a java bean-style setter (.setFooBar) for the given property (:foo-bar) and value."
  [bean prop value]
  (let [setter (->> (str/split (name prop) #"-")
                 (map str/capitalize)
                 (apply str ".set")
                 symbol)]
    ((eval `#(~setter %1 %2)) bean value)))

(defn set-log-level!
  "Sets the global log level for the interal logging system.

   Valid options for `level` are: :OFF, :ERROR, :WARN, :INFO, :DEBUG, :TRACE, :ALL"
  [level]
  (WunderBoss/setLogLevel (name level)))
