(ns dev.jaide.finity.queue
  (:refer-clojure :exclude [loop get]))

(defprotocol ITaskQueue
  "A basic task queue to asynchronously queue actions"

  (enqueue
    [queue task]
    "Append a task to the end of the queue
    
    Arguments:
    - queue - Instance of a queue
    - task - A function to queue
    
    Returns a promise that will be resolved with the task function return value")

  (loop
    [queue]
    "Continuously async loop to drain the queue
    
    Arguments:
    - queue - Instance of a queue
    
    Returns a promise")

  (clear
    [queue]
    "Clear the queue")

  (get
    [queue]
    "Return the queue array"))

(deftype ArrayQueue [queue]
  ITaskQueue

  (enqueue [this task]
    (assert (fn? task) "Queue task must be a function")
    (let [apis (js/Promise.withResolvers)
          promise (.-promise apis)
          resolve (.-resolve apis)]
      (.push queue #(-> (js/Promise.resolve (task))
                        (.then resolve)))
      (loop this)
      promise))

  (loop [this]
    (when-let [task (.pop queue)]
      (-> (task)
          (.then #(when (pos? (.-length queue))
                    (loop this))))))

  (clear [this]
    (.splice queue 0 (.length queue))
    this)

  (get [_this]
    queue))

(defn array-queue
  []
  (let [queue (ArrayQueue. #js [])]
    queue))

(comment
  (let [{:keys [promise resolve reject]} (js/Promise.withResolvers)]
    (println [promise resolve reject])))
