(ns dev.polymeris.sys.db
  (:require [conman.core :as conman]
            [integrant.core :as ig]
            [clojure.tools.logging :as log]
            [clojure.spec.alpha :as s]
            [ragtime.jdbc]
            [ragtime.repl]
            [jsonista.core :as json])
  (:import
    (java.sql Array Date Time Timestamp PreparedStatement)
    (org.postgresql.util PGobject)
    (clojure.lang IPersistentMap IPersistentSet ISeq)))


(s/def ::uri string?)
(s/def ::queries-file string?)
(s/def ::migrations-dir string?)

(s/def ::connection #(not (.isClosed %)))
(s/def ::queries map?)

(s/def ::database
  (s/keys :req-un [::uri ::migrations-dir ::connection ::queries]))

(defn query!
  "Runs a database query and returns the result.
  The query most have been defined in the query-files passed when initializing
  the ::database system."
  [database query-key params]
  (conman/query (:queries database) query-key params))

(s/fdef query!
        :args (s/cat :database ::database
                     :query-key keyword?
                     :params map?))

(defn migration-config
  [{:keys [uri migrations-dir]}]
  {:datastore (ragtime.jdbc/sql-database {:connection-uri uri})
   :migrations (ragtime.jdbc/load-resources migrations-dir)})

(defn migrate!
  [database]
  (-> (migration-config database)
      ragtime.repl/migrate))

(defmethod ig/pre-init-spec ::database [_]
  (s/keys :req-un [::uri]
          :opt-un [::migrations-dir ::queries-file]))

(defmethod ig/init-key ::database
  [_ {:keys [uri migrations-dir queries-file]
      :or {migrations-dir "migrations" queries-file "queries/queries.sql"}}]
  (let [connection (conman/connect! {:jdbc-url uri})
        _ (log/info "Database connection established")
        queries (conman/bind-connection-map connection queries-file)
        _ (log/info "SQL queries loaded")]
    {:uri uri
     :migrations-dir migrations-dir
     :connection connection
     :queries queries}))

(defmethod ig/halt-key! ::database
  [_ {:keys [connection] :as database}]
  (when (s/valid? ::database database)
    (.close connection)
    (log/info "Database connection destroyed")))


(def ^:private json-mapper
  (json/object-mapper
    {:encode-key-fn name
     :decode-key-fn keyword}))

(defn pgobj->clj [^PGobject pgobj]
  (let [type (.getType pgobj)
        value (.getValue pgobj)]
    (case type
      "json" (json/read-value value json-mapper)
      "jsonb" (json/read-value value json-mapper)
      "citext" (str value)
      value)))

(extend-protocol next.jdbc.result-set/ReadableColumn
  Timestamp
  (read-column-by-label [^Timestamp v _]
    (.toLocalDateTime v))
  (read-column-by-index [^Timestamp v _2 _3]
    (.toLocalDateTime v))
  Date
  (read-column-by-label [^Date v _]
    (.toLocalDate v))
  (read-column-by-index [^Date v _2 _3]
    (.toLocalDate v))
  Time
  (read-column-by-label [^Time v _]
    (.toLocalTime v))
  (read-column-by-index [^Time v _2 _3]
    (.toLocalTime v))
  Array
  (read-column-by-label [^Array v _]
    (vec (.getArray v)))
  (read-column-by-index [^Array v _2 _3]
    (vec (.getArray v)))
  PGobject
  (read-column-by-label [^PGobject pgobj _]
    (pgobj->clj pgobj))
  (read-column-by-index [^PGobject pgobj _2 _3]
    (pgobj->clj pgobj)))

(defn clj->jsonb-pgobj [value]
  (doto (PGobject.)
    (.setType "jsonb")
    (.setValue (json/write-value-as-string value json-mapper))))

(extend-protocol next.jdbc.prepare/SettableParameter
  IPersistentMap
  (set-parameter [^IPersistentMap v ^PreparedStatement stmt ^long idx]
    (.setObject stmt idx (clj->jsonb-pgobj v)))
  IPersistentSet
  (set-parameter [^IPersistentMap v ^PreparedStatement stmt ^long idx]
    (.setObject stmt idx (clj->jsonb-pgobj v)))
  ISeq
  (set-parameter [^IPersistentMap v ^PreparedStatement stmt ^long idx]
    (.setObject stmt idx (clj->jsonb-pgobj v))))