(ns vlaaad.reveal.pro.sql.table
  (:require [vlaaad.reveal.view :as view]
            [vlaaad.reveal.style :as style]
            [vlaaad.reveal.event :as event]
            [vlaaad.reveal.pro.sql.connection :as conn]
            [vlaaad.reveal.pro.form.impl :as form.impl]
            [vlaaad.reveal.fx :as rfx]
            [clojure.main :as m]
            [cljfx.ext.tree-view :as fx.ext.tree-view]
            [cljfx.api :as fx]
            [vlaaad.reveal.popup :as popup]
            [vlaaad.reveal.font :as font]
            [clojure.string :as str])
  (:import [javafx.event Event]
           [javafx.scene.control TreeView TreeItem TreeCell]
           [javafx.scene.input KeyEvent KeyCode MouseEvent]
           [javafx.scene Node]))

(defn text-field [props]
  (let [text (:text props)
        width (+ 6 (* (font/char-width) (max 2 (count text))))]
    (assoc props
      :fx/type :text-field
      :style-class "reveal-sql-text-field"
      :min-width width
      :max-width width
      :text text)))

(defmethod event/handle ::update-state [{:keys [id fn]}]
  #(cond-> % (contains? % id) (update id fn)))

(defn- query-table! [id {:keys [conn-fn pattern schema]} handler]
  (let [{:keys [columns] :as parsed} (conn/parse-pattern schema pattern)
        conn (conn/connect conn-fn)
        stmt (.createStatement conn)
        _ (handler {::event/type ::view/create-view-state
                    :id id
                    :state {:columns columns
                            :result ::loading}})
        f (event/daemon-future
            (try
              (let [count (conn/count-parsed-pattern! stmt parsed)]
                (handler
                  {::event/type ::update-state
                   :id id
                   :fn #(assoc % :count count)}))
              (let [items (conn/execute-parsed-pattern! stmt parsed)]
                (handler
                  {::event/type ::update-state
                   :id id
                   :fn #(assoc % :result ::success :items items)}))
              (catch Exception e
                (handler {::event/type ::update-state
                          :id id
                          :fn #(assoc % :result ::failure :exception e)}))
              (finally
                (.close conn))))]

    #(do
       (handler {::event/type ::view/dispose-state :id id})
       (future-cancel f)
       (try (.close stmt) (catch Exception _))
       (try (.close conn) (catch Exception _)))))

(defmethod event/handle ::show-popup [{:keys [on-local-state-changed ^Event fx/event]}]
  (event/handle (assoc on-local-state-changed :fn #(assoc % :node (.getTarget event)))))

(defmethod event/handle ::hide-popup [{:keys [on-local-state-changed]}]
  (event/handle (assoc on-local-state-changed :fn #(dissoc % :node))))

(defmethod event/handle ::mark-expanded [{:keys [on-local-state-changed path fx/event]}]
  (event/handle
    (assoc on-local-state-changed :fn #(update % :expanded (fnil (if event conj disj) #{}) path))))

(defmethod event/handle ::edit-where [{:keys [on-local-state-changed path column text]}]
  (event/handle
    (assoc on-local-state-changed :fn #(assoc % :where-edit {:path path
                                                             :column column
                                                             :text text}))))

(defmethod event/handle ::set-where-text [{:keys [on-local-state-changed fx/event]}]
  (event/handle
    (assoc on-local-state-changed :fn #(assoc-in % [:where-edit :text] event))))

(defmethod event/handle ::set-where [{:keys [on-local-state-changed]}]
  (event/handle
    (assoc on-local-state-changed
      :fn #(let [{:keys [where-edit]} %]
             (-> %
                 (dissoc :where-edit)
                 (cond-> where-edit
                   (update :pattern conn/set-column-clause
                           (:path where-edit)
                           (:column where-edit)
                           :where
                           (not-empty (str/trim (str (:text where-edit)))))))))))

(defmethod event/handle ::set-where-on-blur [{:keys [fx/event] :as e}]
  (if event
    identity
    (event/handle
      (assoc e ::event/type ::set-where))))

(defmethod event/handle ::cancel-where-edit-on-escape [{:keys [^KeyEvent fx/event on-local-state-changed]}]
  (if (= KeyCode/ESCAPE (.getCode event))
    (do
      (.consume event)
      (event/handle (assoc on-local-state-changed :fn #(dissoc % :where-edit))))
    identity))

(defn- where-view-impl [{:keys [text path column local-state on-local-state-changed] :as props}]
  (let [{:keys [where-edit]} local-state]
    (if (and (= path (:path where-edit))
             (= column (:column where-edit)))
      {:fx/type form.impl/ext-with-focus-on-request
       :focused true
       :desc {:fx/type text-field
              :text (str (:text where-edit))
              :on-key-pressed {::event/type ::cancel-where-edit-on-escape
                               :on-local-state-changed on-local-state-changed}
              :on-focused-changed {::event/type ::set-where-on-blur
                                   :on-local-state-changed on-local-state-changed}
              :on-action {::event/type ::set-where
                          :on-local-state-changed on-local-state-changed}
              :on-text-changed {::event/type ::set-where-text
                                :on-local-state-changed on-local-state-changed}}}
      (assoc (if (seq text)
               {:fx/type :label
                :text text}
               {:fx/type :region
                :style-class "reveal-sql-search"})
        :on-mouse-clicked (assoc props ::event/type ::edit-where
                                       :on-local-state-changed on-local-state-changed)))))

(defn- where-view [props]
  {:fx/type fx/ext-get-env
   :env [:local-state :on-local-state-changed]
   :desc (assoc props :fx/type where-view-impl)})

(defn- pattern->tree-items [path schema [{table :table} {:keys [columns joins]}] expanded on-local-state-changed]
  (concat
    (for [col (get-in schema [:tables table :columns])]
      {:fx/type :tree-item
       :value {:text col
               :enabled (contains? columns col)
               :where {:path path
                       :column col
                       :text (get-in columns [col :where])}
               :fn [conn/switch-column path col]}})
    (for [join (get-in schema [:tables table :joins])
          :let [child-pattern (find joins join)
                path (conj path join)]]
      {:fx/type :tree-item
       :value {:text (name (conn/join-kw schema table join))
               :enabled (some? child-pattern)
               :fn [conn/switch-join path (get-in schema [:tables (:table join) :columns])]}
       :expanded (contains? expanded path)
       :on-expanded-changed {::event/type ::mark-expanded
                             :path path
                             :on-local-state-changed on-local-state-changed}
       :children (if (or child-pattern (contains? expanded path))
                   (pattern->tree-items path
                                        schema
                                        (or child-pattern
                                            [join {}])
                                        expanded
                                        on-local-state-changed)
                   [{:fx/type :tree-item
                     :value {:text "Loading..."}}])})))

(defn- describe-pattern-tree-view-cell [{:keys [text enabled where]}]
  (cond-> {:pseudo-classes (if enabled #{:enabled} #{})
           :text text
           :content-display :right}
    where
    (assoc :graphic (assoc where :fx/type where-view))))

(defn- apply-switches [pattern fns]
  (reduce
    (fn [acc [f & args]]
      (apply f acc args))
    pattern
    fns))

(defn- extract-switches [^Event event]
  (->> event
       ^TreeView .getSource
       .getSelectionModel
       .getSelectedItems
       (map #(:fn (.getValue ^TreeItem %)))))

(defmethod event/handle ::filter-tree-view-events [{:keys [^Event fx/event on-local-state-changed]}]
  (cond
    (and (instance? KeyEvent event)
         (= KeyEvent/KEY_PRESSED (.getEventType event)))
    (if (.isShortcutDown ^KeyEvent event)
      (do (when (= (.getSource event) (.getTarget event))
            (.consume event))
          identity)
      (condp = (.getCode ^KeyEvent event)
        KeyCode/SPACE
        (do (.consume event) identity)

        identity))

    (and (instance? MouseEvent event)
         (= MouseEvent/MOUSE_CLICKED (.getEventType event))
         (= 2 (.getClickCount ^MouseEvent event)))
    (do (.consume event)
        (let [fns (extract-switches event)]
          (event/handle (assoc on-local-state-changed :fn #(update % :pattern apply-switches fns)))))

    :else
    identity))

(defmethod event/handle ::handle-tree-view-events [{:keys [^Event fx/event on-local-state-changed]}]
  (cond
    (and (instance? KeyEvent event)
         (= KeyEvent/KEY_PRESSED (.getEventType event)))
    (condp = (.getCode ^KeyEvent event)
      KeyCode/ENTER
      (do (.consume event)
          (let [fns (extract-switches event)]
            (event/handle
              (assoc on-local-state-changed :fn #(update % :pattern apply-switches fns)))))

      KeyCode/ESCAPE
      (do (.consume event)
          (event/handle {::event/type ::hide-popup
                         :on-local-state-changed on-local-state-changed}))

      KeyCode/SLASH
      (let [tree ^TreeView (.getSource event)
            where (-> tree
                      .getFocusModel
                      ^TreeItem .getFocusedItem
                      .getValue
                      :where)]
        (if where
          (do (.consume event)
              (event/handle
                (assoc where
                  ::event/type ::edit-where
                  :on-local-state-changed on-local-state-changed)))
          identity))

      identity)

    (and (instance? MouseEvent event)
         (= MouseEvent/MOUSE_CLICKED (.getEventType event))
         (= 2 (.getClickCount ^MouseEvent event)))
    (do (.consume event)
        (let [fns (extract-switches event)]
          (event/handle (assoc on-local-state-changed :fn #(update % :pattern apply-switches fns)))))

    :else
    identity))

(defn- pattern-tree-view [{:keys [schema local-state on-local-state-changed]}]
  (let [{:keys [pattern expanded]} local-state]
    {:fx/type fx.ext.tree-view/with-selection-props
     :props {:selected-index 0
             :selection-mode :multiple}
     :desc {:fx/type :tree-view
            :show-root false
            :max-height 350
            :event-filter {::event/type ::filter-tree-view-events
                           :on-local-state-changed on-local-state-changed}
            :event-handler {::event/type ::handle-tree-view-events
                            :on-local-state-changed on-local-state-changed}
            :cell-factory {:fx/cell-type :tree-cell
                           :describe describe-pattern-tree-view-cell}
            :root {:fx/type :tree-item
                   :children (pattern->tree-items [] schema pattern expanded on-local-state-changed)}}}))

(defn- pattern-view [{:keys [schema local-state on-local-state-changed]}]
  (let [{:keys [^Node node]} local-state]
    {:fx/type fx/ext-let-refs
     :refs (when node
             {::popup {:fx/type popup/view
                       :width 250
                       :bounds (.localToScreen node (.getBoundsInLocal node))
                       :window (.getWindow (.getScene node))
                       :on-cancel {::event/type ::hide-popup
                                   :on-local-state-changed on-local-state-changed}
                       :position :top
                       :desc {:fx/type :v-box
                              :padding style/default-padding
                              :children [{:fx/type pattern-tree-view
                                          :schema schema
                                          :local-state local-state
                                          :on-local-state-changed on-local-state-changed}]}}})
     :desc {:fx/type :v-box
            :children [{:fx/type :button
                        :text "Edit query"
                        :on-action {::event/type ::show-popup
                                    :on-local-state-changed on-local-state-changed}}]}}))

(defmethod event/handle ::set-limit-text [{:keys [on-local-state-changed fx/event]}]
  (event/handle (assoc on-local-state-changed :fn #(assoc % :limit-text event))))

(defmethod event/handle ::set-limit [{:keys [on-local-state-changed limit]}]
  (event/handle (assoc on-local-state-changed :fn #(assoc-in % [:pattern 0 :limit] limit))))

(defn- limit-input [{:keys [local-state on-local-state-changed]}]
  (let [{:keys [pattern limit-text]} local-state
        existing-limit (get-in pattern [0 :limit])
        text (str (or limit-text existing-limit))
        new-limit (try
                    (let [n (Integer/parseInt text)]
                      (when (pos-int? n) n))
                    (catch Exception _))]
    (cond->
      {:fx/type text-field
       :pseudo-classes (cond
                         (not new-limit) #{:error}
                         (not= new-limit existing-limit) #{:active}
                         :else #{})
       :text text
       :on-text-changed {::event/type ::set-limit-text
                         :on-local-state-changed on-local-state-changed}}

      new-limit
      (assoc :on-action {::event/type ::set-limit
                         :limit new-limit
                         :on-local-state-changed on-local-state-changed}))))


(defn- table-explore-view-impl-impl [{:keys [columns
                                             result
                                             exception
                                             items
                                             count
                                             schema]
                                      :or {items []}}]
  {:fx/type :v-box
   :children [{:fx/type view/table
               :v-box/vgrow :always
               :columns columns
               :placeholder (case result
                              ::loading {:fx/type :label :text "Loading..."}
                              ::failure {:fx/type :label
                                         :style {:-fx-text-fill (style/color :error)}
                                         :wrap-text true
                                         :text (-> exception
                                                   Throwable->map
                                                   m/ex-triage
                                                   m/ex-str)}
                              {:fx/type :label :text ""})
               :items items}
              {:fx/type :h-box
               :alignment :center-left
               :children [{:fx/type fx/ext-get-ref
                           :ref ::pattern-view}
                          {:fx/type :region
                           :h-box/hgrow :always}
                          (if (:can-limit schema)
                            {:fx/type :h-box
                             :padding {:right style/default-padding}
                             :alignment :center-left
                             :children [{:fx/type :label
                                         :text (str (or count "loading") " rows, limit: ")}
                                        {:fx/type fx/ext-get-ref
                                         :ref ::limit-input}]}
                            {:fx/type :label
                             :text (str (or count "loading") " rows")})]}]})

(defn- table-explore-view-impl [{:keys [conn-fn schema local-state on-local-state-changed]}]
  {:fx/type fx/ext-set-env
   :env {:local-state local-state
         :on-local-state-changed on-local-state-changed}
   :desc
   {:fx/type fx/ext-let-refs
    :refs {::pattern-view {:fx/type pattern-view
                           :schema schema
                           :local-state local-state
                           :on-local-state-changed on-local-state-changed}
           ::limit-input {:fx/type limit-input
                          :local-state local-state
                          :on-local-state-changed on-local-state-changed}}
    :desc
    {:fx/type rfx/ext-with-process
     :args {:conn-fn conn-fn
            :pattern (:pattern local-state)
            :schema schema}
     :start query-table!
     :desc {:fx/type table-explore-view-impl-impl
            :schema schema
            :local-state local-state
            :on-local-state-changed on-local-state-changed}}}})

(defn view [props]
  {:fx/type rfx/ext-with-local-state
   :initial-state {:pattern (:pattern props)}
   :desc (-> props
             (dissoc :pattern)
             (assoc :fx/type table-explore-view-impl))})