(ns vlaaad.reveal.pro.form.state
  "© 2021 Vladislav Protsenko. All rights reserved."
  (:require [vlaaad.reveal.pro.form.impl :as impl]
            [vlaaad.reveal.fx :as rfx]
            [vlaaad.reveal.event :as event])
  (:import [clojure.lang IRef]))

(defprotocol IEditState
  (edit-state [state] "Returns edit state")
  (edit-form [state] "Returns edit form"))

(defn- edit->result [{:keys [explain]} edit]
  (let [result (impl/validate explain edit)]
    (if-let [error (impl/error result)]
      {:error error}
      {:value (impl/value result)})))

(deftype State [form atom]
  IEditState
  (edit-state [_] atom)
  (edit-form [_] form)
  IRef
  (deref [_] (edit->result form @atom))
  (addWatch [this key callback]
    (add-watch atom [this key] #(callback key this (edit->result form %3) (edit->result form %4))))
  (removeWatch [this key]
    (remove-watch atom [this key])))

(defn state [form value]
  (->State form (atom (impl/edit (:editor form) value))))

(defmethod event/handle ::init-form [{:keys [id edit]}]
  #(assoc % id {:id id :edit edit}))

(defmethod event/handle ::dispose-form [{:keys [id]}]
  #(dissoc % id))

(defmethod event/handle ::edit [{:keys [fn atom]}]
  (swap! atom fn)
  identity)

(defn- state-view-impl [{:keys [edit atom form]}]
  {:fx/type impl/form-view
   :form form
   :edit edit
   :on-edit {::event/type ::edit :atom atom}})

(defn- init-form! [id atom handler]
  (handler {::event/type ::init-form :id id :edit @atom})
  (let [key [`init-form! id]]
    (add-watch atom key #(handler {::event/type ::init-form :id id :edit %4}))
    #(do (remove-watch atom key)
         (handler {::event/type ::dispose-form :id id}))))

(defn view
  "Cljfx component fn that shows stateful form view"
  [{:keys [state]}]
  (let [a (edit-state state)]
    {:fx/type :scroll-pane
     :style-class "reveal-form"
     :focus-traversable false
     :content {:fx/type rfx/ext-with-process
               :start init-form!
               :args a
               :desc {:fx/type state-view-impl
                      :form (edit-form state)
                      :atom a}}}))