(ns vlaaad.reveal.pro.form.coll-of
  "© 2021 Vladislav Protsenko. All rights reserved."
  (:require [vlaaad.reveal.pro.form.impl :as impl]
            [vlaaad.reveal.pro.form.compatible :as compatible]
            [vlaaad.reveal.pro.form.copy-paste :as copy-paste]
            [vlaaad.reveal.pro.form.vignette :as vignette]
            [vlaaad.reveal.event :as event]
            [vlaaad.reveal.font :as font])
  (:import [javafx.scene.input KeyCode KeyEvent]
           [javafx.scene.control ComboBox]
           [java.util UUID]))

(defmethod event/handle ::reorder [{:keys [index direction on-edit]}]
  (let [swap-index (direction index)]
    (event/handle (assoc on-edit :fn (fn [edit]
                                       (update edit :order-ids #(assoc % index (% swap-index)
                                                                         swap-index (% index))))))))

(defmethod event/handle ::delete [{:keys [index on-edit focus-gen]}]
  (event/handle
    (assoc on-edit
      :fn (fn [edit]
            (let [{:keys [order-ids]} edit
                  id (order-ids index)]
              (-> edit
                  (assoc :focus [(or
                                   (and (< (inc index) (count order-ids))
                                        (order-ids (inc index)))
                                   (and (pos? index)
                                        (order-ids (dec index)))
                                   ::add)
                                 (inc focus-gen)])
                  (assoc :order-ids (into (subvec order-ids 0 index) (subvec order-ids (inc index))))
                  (update :order-id->edit dissoc id)))))))


(defmethod event/handle ::edit-item [{:keys [fn on-edit id]}]
  (event/handle (assoc on-edit :fn #(update-in % [:order-id->edit id] fn))))

(defmethod event/handle ::add-item [{:keys [on-edit item-form focus-gen]}]
  (let [id (UUID/randomUUID)]
    (event/handle (assoc on-edit :fn #(-> %
                                          (update :order-ids conj id)
                                          (assoc :focus [id (inc focus-gen)])
                                          (assoc-in [:order-id->edit id] (impl/edit (:editor item-form))))))))

(defn- edit-content [item-form min-count coll]
  (let [coll (if (impl/completely-undefined? coll)
               (repeat min-count impl/undefined)
               coll)
        order (vec (range (count coll)))]
    {:order-ids order
     :focus [nil 0]
     :order-id->edit (zipmap order
                             (map #(impl/edit (:editor item-form) %) coll))}))

(defn- assemble-content [{:keys [order-ids order-id->edit]}]
  (impl/assemble-all (mapv order-id->edit order-ids)))

(defn- view-content [item-form max-count {:keys [edit on-edit]}]
  (let [{:keys [order-ids order-id->edit focus]} edit
        [focus-id focus-gen] focus]
    {:fx/type impl/multi-line-view
     :children (cond->
                 (vec (for [[index id] (map-indexed vector order-ids)
                            :let [item-edit (order-id->edit id)]]
                        {:fx/type impl/ext-with-focus-on-request
                         :fx/key id
                         :focused (when (= id focus-id) focus-gen)
                         :desc
                         {:fx/type impl/form-view
                          :edit item-edit
                          :on-edit {::event/type ::edit-item
                                    :id id
                                    :on-edit on-edit}
                          :form (update item-form :options
                                        #(-> %
                                             (or [])
                                             (cond->
                                               (pos? index)
                                               (conj {:invoke {::event/type ::reorder
                                                               :index index
                                                               :direction dec
                                                               :on-edit on-edit}
                                                      :label "Move Item Up"
                                                      :shortcut [:alt :up]})

                                               (< index (dec (count order-ids)))
                                               (conj {:invoke {::event/type ::reorder
                                                               :index index
                                                               :direction inc
                                                               :on-edit on-edit}
                                                      :label "Move Item Down"
                                                      :shortcut [:alt :down]}))
                                             (conj {:invoke {::event/type ::delete
                                                             :focus-gen focus-gen
                                                             :index index
                                                             :on-edit on-edit}
                                                    :label "Delete Item"
                                                    :shortcut [:alt :back-space]})))}}))
                 (< (count order-ids) max-count)
                 (conj {:fx/type impl/ext-with-focus-on-request
                        :focused (when (= ::add focus-id) focus-gen)
                        :desc {:fx/type :button
                               :style-class "reveal-form-button"
                               :on-action {::event/type ::add-item
                                           :focus-gen focus-gen
                                           :item-form item-form
                                           :on-edit on-edit}
                               :text (str "+ " (:label item-form))}}))}))

(defn content-editor
  "edits any collections as vectors"
  [{:keys [item-form min-count max-count]
    :or {min-count 0
         max-count Integer/MAX_VALUE}}]
  (impl/make-editor
    :edit #(edit-content item-form min-count %)
    :assemble assemble-content
    :view #(view-content item-form max-count %)))

(defmethod event/handle ::set-kind [{:keys [kind on-edit]}]
  (event/handle (assoc on-edit :fn #(assoc % :kind kind))))

(defmethod event/handle ::edit-content [{:keys [on-edit fn]}]
  (event/handle (assoc on-edit :fn #(update % :content fn))))

(defn- view [kinds {:keys [edit on-edit form]}]
  (let [{:keys [kind content]} edit]
    {:fx/type impl/multi-line-view
     :children
     [{:fx/type vignette/view
       :edit edit
       :on-edit on-edit
       :form form
       :desc {:fx/type impl/menu-button
              :pseudo-classes #{:object}
              :text (case kind
                      :map "{"
                      :list "("
                      :vector "["
                      :set "#{")
              :items (for [k kinds]
                       {:fx/type :menu-item
                        :text (pr-str k)
                        :on-action {::event/type ::set-kind
                                    :on-edit on-edit
                                    :kind k}})}}
      {:fx/type impl/indent-view
       :desc {:fx/type impl/content-view
              :edit content
              :on-edit {::event/type ::edit-content
                        :on-edit on-edit}}}
      {:fx/type :label
       :style-class "reveal-form-object"
       :text (case kind
               :map "}"
               :list ")"
               :vector "]"
               :set "}")}]}))

(defn- get-kind [x]
  (cond
    (map? x) :map
    (vector? x) :vector
    (set? x) :set
    (sequential? x) :list))

(defn- edit [kinds content-editor coll]
  {:kind (if (impl/completely-undefined? coll)
           (first kinds)
           (get-kind coll))
   :content (impl/edit content-editor coll)})

(defn- assemble [{:keys [kind content]}]
  (impl/fmap-result
    (impl/assemble content)
    (fn [items]
      (case kind
        :map (into {} items)
        :list (apply list items)
        :vector items
        :set (set items)))))

(defn value-editor [{:keys [kinds item-form min-count max-count]
                     :or {kinds [:vector :set :map :list]}
                     :as opts}]
  (let [content-editor (content-editor opts)]
    (-> (impl/make-editor
          :edit #(edit kinds content-editor %)
          :assemble assemble
          :view #(view kinds %))
        (compatible/wrap #(some? (get-kind %)))
        (copy-paste/wrap))))
