(ns vlaaad.reveal.pro.vega
  (:require [vlaaad.reveal.vega :as vega]
            [vlaaad.reveal.view :as view]
            [vlaaad.reveal.event :as event]
            [vlaaad.reveal.ui :as ui])
  (:import [java.util.concurrent TimeUnit]
           [clojure.lang PersistentQueue]
           [java.lang.management ManagementFactory]
           [java.time Instant]
           [com.sun.management OperatingSystemMXBean]
           [java.time.temporal ChronoUnit]))

(defn resources-view [{:keys [seconds cpu memory]
                       :or {seconds 300
                            cpu true
                            memory true}}]
  (let [seconds (-> seconds (max 10) (min 28800))]
    {:fx/type view/observable-view
     :subscribe (fn [notify]
                  (let [state (atom (into PersistentQueue/EMPTY
                                          (->> (.minus (Instant/now) seconds ChronoUnit/SECONDS)
                                               (iterate #(.plus ^Instant % 1 ChronoUnit/SECONDS))
                                               (take seconds)
                                               (map #(hash-map :time (.toEpochMilli ^Instant %))))))

                        runtime (Runtime/getRuntime)
                        add (fn [q v]
                              (pop (conj q v)))
                        f (.scheduleAtFixedRate event/daemon-scheduler
                                                #(notify (swap! state add
                                                                (cond-> {:time (.toEpochMilli (Instant/now))}
                                                                  cpu
                                                                  (assoc :cpu (let [load (.getProcessCpuLoad ^OperatingSystemMXBean (ManagementFactory/getOperatingSystemMXBean))]
                                                                                (when-not (neg? load) load)))
                                                                  memory
                                                                  (assoc :total (.totalMemory runtime)
                                                                         :free (.freeMemory runtime)))))
                                                0 1 TimeUnit/SECONDS)]
                    #(future-cancel f)))
     :fn (fn [data]
           {:fx/type vega/view
            :data data
            :spec {:encoding {:x {:field :time
                                  :type :temporal
                                  :title false}}
                   :layer (cond-> []
                            memory
                            (conj {:transform [{:calculate "datum.total - datum.free"
                                                :as :used}]
                                   :mark {:type :area :fill "#4285F4"}
                                   :encoding {:y {:field :used
                                                  :type :quantitative
                                                  :stack false
                                                  :title "Used memory"
                                                  :axis {:format "~s" :titleColor "#4285F4"}}}})
                            cpu
                            (conj {:mark {:type :line :stroke "#FF7043"}
                                   :encoding {:y {:field :cpu
                                                  :title "CPU"
                                                  :type :quantitative
                                                  :axis {:format "%" :titleColor "#FF7043"}
                                                  :scale {:domain [0 1]}}}}))
                   :resolve {:scale {:y :independent}}}})}))

(defn resources-sticker [opts]
  (ui/sticker (merge {:fx/type resources-view}
                     (select-keys opts [:seconds :cpu :memory]))
              (merge {:title "JVM resources"
                      :default-size {:width 260 :height 125}}
                     (dissoc opts :seconds :cpu :memory))))