(ns vlaaad.reveal.pro.form.text
  "© 2021 Vladislav Protsenko. All rights reserved."
  (:require [clojure.string :as str]
            [vlaaad.reveal.event :as event]
            [vlaaad.reveal.font :as font]
            [vlaaad.reveal.pro.form.vignette :as vignette]
            [vlaaad.reveal.pro.form.impl :as impl]
            [vlaaad.reveal.fx :as rfx])
  (:import [clojure.lang LineNumberingPushbackReader]
           [java.io StringReader]
           [javafx.scene.input KeyEvent KeyCode]
           [javafx.event Event]))

(defmethod event/handle ::edit-text [{:keys [fx/event on-edit]}]
  (event/handle (assoc on-edit :fn #(assoc % :text event))))

(defn- filter-text-area-events [e]
  (when (instance? KeyEvent e)
    (let [^KeyEvent e e]
      (when (and (= KeyEvent/KEY_PRESSED (.getEventType e))
                 (= KeyCode/TAB (.getCode e))
                 (not (.isControlDown e)))
        (.consume e)
        (Event/fireEvent
          (.getTarget e)
          (KeyEvent.
            (.getSource e)
            (.getTarget e)
            (.getEventType e)
            (.getCharacter e)
            (.getText e)
            (.getCode e)
            (.isShiftDown e)
            true
            (.isAltDown e)
            (.isMetaDown e)))))))

(defn- view [{:keys [edit on-edit form]}]
  (let [{:keys [text]} edit
        width (-> (apply max (or (seq (map count (str/split-lines text))) [0]))
                  (max 2)
                  (* (font/char-width)))]
    {:fx/type vignette/view
     :edit edit
     :on-edit on-edit
     :form form
     :desc {:fx/type :text-area
            :style-class "reveal-form-text-input"
            :min-width width
            :max-width width
            :min-height 1
            :pref-height (inc (* (inc (font/line-height)) (inc (count (filter #{\newline} text)))))
            :text-formatter rfx/code-text-formatter
            :text text
            :event-filter filter-text-area-events
            :on-text-changed {::event/type ::edit-text :on-edit on-edit}}}))

(defn- edit [x]
  {:text (if (impl/completely-undefined? x) "" (impl/pprint-str x))})

(defn- assemble [{:keys [text]}]
  (let [reader (LineNumberingPushbackReader. (StringReader. text))
        value (read {:read-cond :allow :eof impl/undefined} reader)
        empty (Object.)
        extra (read {:read-cond :allow :eof empty} reader)]
    (if (identical? empty extra)
      value
      (impl/error-result "too many forms"))))

(def value-editor
  (impl/make-editor
    :edit edit
    :assemble assemble
    :view view))

(defn make-edit-with-text [text]
  (assoc (impl/edit value-editor) :text text))