(ns vlaaad.reveal.pro.sql.schema-graph
  (:require [vlaaad.reveal.stream :as stream]
            [vlaaad.reveal.pro.sql.table :as table]
            [clojure.string :as str]
            [vlaaad.reveal.event :as event]
            [cljfx.ext.web-view :as fx.ext.web-view]
            [vlaaad.reveal.style :as style]
            [cljfx.api :as fx]
            [clojure.java.io :as io]
            [cljfx.fx.web-view :as fx.web-view])
  (:import [javafx.scene.web WebEvent]))


(defmethod event/handle ::explore-table [{:keys [^WebEvent fx/event view-id view-index schema conn-fn]}]
  (let [{:keys [table cmd shift]} (read-string (.getData event))]
    (event/handle {::event/type :vlaaad.reveal.ui/view
                   :value {:fx/type table/view
                           :conn-fn conn-fn
                           :pattern [{:table table
                                      :limit 10000}
                                     {:columns (zipmap (get-in schema [:tables table :columns])
                                                       (repeat {}))
                                      :joins {}}]
                           :schema schema}
                   :form (stream/horizontal
                           (stream/raw-string "(" {:fill :util})
                           (stream/raw-string "db:explore-table " {:fill :symbol})
                           (stream/raw-string (str/join "." table) {:fill :object})
                           (stream/raw-string ")" {:fill :util}))
                   :target (cond
                             shift :inspector
                             cmd :new-result-panel)
                   :view-index view-index
                   :view-id view-id})))

(defn render-dot [statements]
  (letfn [(render-id [id]
            (let [parts (if (sequential? id) id [id])]
              (->> parts
                   (map #(if (keyword? %) (name %) (pr-str %)))
                   (str/join ":"))))
          (render-html [[tag & tail]]
            (let [attrs (if (map? (first tail)) (first tail) {})
                  children (if (map? (first tail)) (rest tail) tail)]
              (str "<" (name tag)
                   " "
                   (->> attrs
                        (map (fn [[k v]]
                               (str (name k)
                                    "="
                                    (pr-str (cond
                                              (keyword? v) (name v)
                                              :else (str v))))))
                        (str/join " "))
                   ">"
                   (->> children
                        (map (fn [child]
                               (if (vector? child) (render-html child) (str child))))
                        str/join)
                   "</" (name tag) ">")))
          (render-attrs [attrs sep]
            (->> attrs
                 (map (fn [[k v]]
                        (str (name k)
                             "="
                             (cond
                               (keyword? v) (name v)
                               (vector? v) (str "<" (render-html v) ">")
                               :else (pr-str v)))))
                 (str/join sep)))
          (render-statements [statements]
            (->> statements
                 (map (fn [stmt]
                        (if (map? stmt)
                          (render-attrs stmt ";")
                          (case (first stmt)
                            :node (let [[_ id attrs] stmt]
                                    (str (render-id id)
                                         (when attrs
                                           (str " ["
                                                (render-attrs attrs ",")
                                                "]"))))
                            :edge (let [[_ from to attrs] stmt]
                                    (str (render-id from)
                                         " -> "
                                         (render-id to)
                                         (when attrs
                                           (str " ["
                                                (render-attrs attrs ",")
                                                "]"))))
                            :subgraph (let [[_ name & statements] stmt]
                                        (str "subgraph " (pr-str (str name)) "{"
                                             (render-statements statements)
                                             "}"))))))
                 (str/join ";")))]
    (str "digraph {"
         (render-statements statements)
         "}")))

(defn graph-view-impl [{:keys [schema statements conn-fn view-id view-index]}]
  {:fx/type fx.ext.web-view/with-engine-props
   :props
   {:on-alert {::event/type ::explore-table
               :view-id view-id
               :view-index view-index
               :schema schema
               :conn-fn conn-fn}
    :content
    (format
      "<head>
           <script src=\"%s\"></script>
           <script src=\"%s\"></script>
           <style>
             body {background-color:%s;margin:0px}
             .node a:focus polygon {fill: %s}
           </style>
         </head>
         <body>
           <script>
             new Viz().renderSVGElement(%s).then(function(element) {
               document.body.appendChild(element);
               var nodes = element.querySelectorAll('.node');
               for (var i = 0; i < nodes.length; i++) {
                 var x = nodes[i];
                 x.addEventListener('click', function (e) {
                   alert('{:table ' + e.currentTarget.id + ' :cmd ' + (e.ctrlKey || e.metaKey) + ' :shift ' + e.shiftKey + '}');
                 });
               }
             });
           </script>
         </body>"
      (.toExternalForm (io/resource "vlaaad/reveal/pro/sql/viz.js"))
      (.toExternalForm (io/resource "vlaaad/reveal/pro/sql/full.render.js"))
      @style/background-color
      @style/selection-color
      (pr-str (render-dot statements)))}
   :desc {:fx/type fx.web-view/lifecycle
          :context-menu-enabled false}})

(defn graph-view [props]
  {:fx/type fx/ext-get-env
   :env {:vlaaad.reveal.ui/id :view-id :vlaaad.reveal.ui/index :view-index}
   :desc (assoc props :fx/type graph-view-impl)})

(defn- table-levels->nodes [tables]
  (for [[k vs] (group-by ffirst tables)
        :let [subgraph (-> vs ffirst rest seq)]
        s (if subgraph
            [(into [:subgraph (str "cluster_" k)
                    {:label k
                     :color @style/popup-color
                     :bgcolor (str @style/popup-color "88")}]
                   (table-levels->nodes
                     (map (fn [[k v]] [(rest k) v]) vs)))]
            (for [[_ {:keys [name columns pk]}] vs]
              [:node (pr-str name)
               {:id (pr-str name)
                :tooltip (last name)
                :label (into [:table
                              {:border 0
                               :cellborder 1
                               :cellspacing 0
                               :href "#"
                               :bgcolor @style/popup-color}
                              [:tr [:td
                                    {:bgcolor @style/unfocused-selection-color
                                     :align :left}
                                    [:font {:color (style/color :object)}
                                     (last name)]]]]
                             (for [c columns]
                               [:tr
                                [:td {:port c :align :left}
                                 (if (some #{c} pk)
                                   [:font {:color (style/color :scalar)}
                                    c]
                                   c)]]))}]))]
    s))

(defn view [{:keys [schema conn-fn]}]
  (let [{:keys [tables url catalog]} schema]
    {:fx/type graph-view
     :schema schema
     :conn-fn conn-fn
     :statements (concat
                   [[:node :node {:shape :plaintext
                                  :fontsize 10
                                  :fontcolor (style/color :symbol)
                                  :color @style/background-color}]
                    [:node :edge {:color (style/color :util)
                                  :arrowsize 0.75}]
                    [:node :graph {:rankdir :LR
                                   :nodesep 0.1
                                   :ranksep 0.25
                                   :pack true
                                   :bgcolor @style/background-color
                                   :fontcolor (style/color :symbol)
                                   :label (or catalog url)
                                   :labelloc "t"}]]
                   (table-levels->nodes tables)
                   (for [[from {:keys [fks]}] tables
                         {:keys [table src dst]} fks
                         [from-port port] (mapv vector src dst)]
                     [:edge [(pr-str from) from-port] [(pr-str table) port]]))}))