(ns om-charts.core
  (:require [om.core :as om]
            [om-tools.dom :refer [svg rect g div strong]]
            [goog.string :as gstring]
            [goog.string.format]))

(defn maximum-value [data]
  (transduce (comp (mapcat :bars)
                   (map :value))
             max 0 data))

(defn inc-offset! [offset increment]
  (let [old-offset @offset]
    (vswap! offset + increment)
    old-offset))

(defn group-offsets! [n offset increment last-increment]
  (let [offsets (doall (repeatedly n #(inc-offset! offset increment)))]
    (vswap! offset + last-increment)
    offsets))

(defn bar-positions [settings groups]
  (let [offset (volatile! 0)
        group-offset (get-in settings [:groups :offset])
        bar-spacing (+ (get-in settings [:bars :width])
                       (get-in settings [:bars :offset]))]
    (mapcat
      (fn [group]
        (group-offsets! (count (:bars group))
                        offset
                        bar-spacing
                        group-offset))
      groups)))

(defn bar-heights [context groups]
  (let [max-val (maximum-value groups)
        context-height (:height context)]
    (sequence (comp (mapcat :bars)
                    (map :value)
                    (map #(* (/ % max-val) context-height)))
              groups)))

(defn render-default-tooltip [settings tooltip]
  (let [bounding-rect (.getBoundingClientRect (:target tooltip))]
    (div {:style (merge {:position         "absolute"
                         :padding          "3px"
                         :background-color "grey"
                         :color            "white"
                         :top              (- (:y tooltip) 10)
                         :left             (.-right bounding-rect)}
                        (get-in settings [:tooltips :style]))}

         (get-in tooltip [:bar :label])
         ": "
         (strong {} (gstring/format "%.2f" (get-in tooltip [:bar :value]))))))

(def default-state
  {:settings {:bars       {:width  50
                           :offset 1
                           :style {:fill "#2196F3"}}
              :groups     {:offset 10}
              :dimensions {:width  400
                           :height 150}
              :tooltips   {:enabled true
                           :render-fn render-default-tooltip
                           :style     {:background-color "#37474F"
                                       :color "#ECEFF1"
                                       :border-radius "3px"
                                       :pointer-events "none"}}}})

  ;; Data: [{:label str :bars [{:label str :value num}]}]
(defn bar-chart [groups owner]
  (reify
    om/IDisplayName
    (display-name [_]
      "Bar chart")
    om/IInitState
    (init-state [_]
      {:active-tooltip nil})
    om/IRender
    (render [_]
      (let [settings (merge (get-in default-state [:settings])
                            (om/get-state owner :settings))
            bar-positions (bar-positions settings groups)
            bar-heights (bar-heights (:dimensions settings) groups)
            tooltips-enabled (get-in settings [:tooltips :enabled])]
        (div {}
             (svg (get-in settings [:dimensions])
               (for [[bar bar-position bar-height] (map vector (mapcat :bars groups) bar-positions bar-heights)]
                 (rect {:y              (- (get-in settings [:dimensions :height]) bar-height)
                        :x              bar-position
                        :height         bar-height
                        :width          (get-in settings [:bars :width])
                        :style          (get-in settings [:bars :style])
                        :on-mouse-move  (when tooltips-enabled
                                          (fn [e]
                                            (om/set-state! owner :active-tooltip {:bar    bar
                                                                                  :target (.-currentTarget e)
                                                                                  :x      (.-clientX e)
                                                                                  :y      (.-clientY e)})))
                        :on-mouse-leave (when tooltips-enabled
                                          (fn [_]
                                            (om/set-state! owner :active-tooltip nil)))})))
             (when-let [tooltip (om/get-state owner :active-tooltip)]
               ((get-in settings [:tooltips :render-fn])
                 settings tooltip)))))))
