(ns discord.extensions.karma
  (:require [clojure.string :as s]
            [taoensso.timbre :as timbre]
            [discord.bot :as bot]
            [discord.config :as config]
            [discord.embeds :as embed]
            [discord.permissions :as perm]))

;;; Create the configuraton file if necessary
(defonce karma-file "data/karma/karma.json")
(config/file-io karma-file :check)

;;; This is where we'll maintain the user's karma
(defonce user-karma
  (atom (config/file-io karma-file :load :default {} :key-fn str)))

;;; This is the pattern that messages much match to alter the karma in the channel
(defonce karma-message-pattern
  (re-pattern "^<@(?<user>\\d+)>\\s*[+-](?<deltas>[+-]+)\\s*"))

(bot/defextension karma [client message]
  (:get
    "Retrieve the karma for mentioned users."
    (let [channel (-> message :channel :id str)
          users   (map :id (:user-mentions message))
          karmas  (for [user users]
                    (format "<@%s>: %s" user (get-in @user-karma [channel user] 0)))]
      (bot/say (format "Karma: \n%s" (s/join \newline karmas)))))

  (:top
    "Retrieves the karma of the top 10 users."
    (let [channel       (-> message :channel :id str)
          users         (take 10 (sort-by val > (get @user-karma channel {})))
          message-body  (for [[user karma] users] (format "<@%s>: %s" (name user) karma))]
      (if (not-empty users)
        (bot/say (format "Highest karma users:\n%s" (s/join \newline message-body)))
        (bot/say "There are no users with karma in this guild!"))))

  (:bottom
    "Retrieves the karma of the bottom 10 users."
    (let [channel       (-> message :channel :id str)
          users         (take 10 (sort-by val < (get @user-karma channel {})))
          message-body  (for [[user karma] users] (format "<@%s>: %s" (name user) karma))]
      (if (not-empty users)
        (bot/say (format "Lowest karma users:\n%s" (s/join \newline message-body)))
        (bot/say "There are no users with karma in this guild!"))))

  (:clear
    "Clear the karma of all users in the channel."
    {:requires [perm/ADMINISTRATOR]}
    (reset! user-karma {})
    (config/file-io karma-file :save :data @user-karma)
    (bot/say "I've cleared out karma across all servers.")))

(defn- update-karma! [channel user karma-delta]
  (let [current-karma (get-in @user-karma [channel user] 0)
        updated-karma (+ current-karma karma-delta)]
    (swap! user-karma assoc-in [channel user] updated-karma)
    (config/file-io karma-file :save :data @user-karma)
    updated-karma))

(defn- calculate-karma-delta
  "Given a string of '+' and '-' characters, determines the overall karma impact."
  [delta-string]
  (->> delta-string
       (map {\+ 1 \- -1})
       (reduce + 0)))

(bot/defhandler karma-message-handler [prefix client message]
  (if-let [[match user-id deltas] (re-find karma-message-pattern (:content message))]
    (let [channel           (-> message :channel :id str)
          author            (-> message :author :id str)
          user-karma-delta  (calculate-karma-delta deltas)]
      (if (= author user-id)
        (bot/say "Nice try, buddy! You can't change your own karma :smiley:")
        (let [new-user-karma (update-karma! channel user-id user-karma-delta)]
          (bot/say (format "Updating <@%s>'s karma to %s" user-id new-user-karma)))))))
