(ns discord.extensions.builtin.groups
  (:require
    [clojure.string :as s]
    [discord.api.guilds :as guilds-api]
    [discord.interactions.core :as i]
    [discord.interactions.components :as comps]
    [discord.interactions.slash :as slash]
    ))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Describing and registering the slash commands
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(slash/register-globally-on-startup!
  (slash/command
    :groups "Server group management."
    (slash/sub-command-group
      :settings "Settings for groups within your server."
      (slash/sub-command
        :create-group-channels "Toggle whether new groups auto-create group specific channels"
        (slash/boolean-option
          :enabled "If true, a text channel is auto-created for a new group."
          :required? true)))
    (slash/sub-command :init "Initialize the group functionality within this guild.")
    (slash/sub-command :list "List available groups to join")
    (slash/sub-command
      :create "Create a new group"
      (slash/string-option :name "The name of the group being created." :required? true)
      (slash/string-option :description "Tell us about your group!" :required? true))
    (slash/sub-command
      :join "Join a specific group or see what groups are available to join via a menu!"
      (slash/string-option :group "The name of the group you want to join."))
    (slash/sub-command
      :modify "Modify an existing group"
      (slash/string-option :group "The group you wish to modify." :required? true)
      (slash/string-option
        :new-name "The new name for the group, if it's being changed.")
      (slash/string-option
        :new-description "The new description for the group, if it's being changed."))
    (slash/sub-command
      :leave "Leave a group"
      (slash/string-option :name "The name of the group you wish to leave." :required? true))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Storing the data for the groups
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defrecord Group [group-name description role-id])

(def groups-by-guild
  (atom {}))

(defn get-group
  [guild-id group-name]
  (get-in @groups-by-guild [guild-id group-name]))

(defn group-exists?
  [guild-id group-name]
  (some? (get-group guild-id group-name)))

(defn add-group-to-guild!
  [auth guild-id group-name description]
  (when-not (group-exists? guild-id group-name)
    (let [role (guilds-api/create-role auth guild-id group-name)
          group (->Group group-name description (:id role))]
      (swap! groups-by-guild assoc-in [guild-id group-name] group))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Administration and server setup
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmethod slash/handle-slash-command-interaction [:groups :init]
  [{:keys [interaction]} auth metadata])

(defmethod slash/handle-slash-command-interaction [:groups :settings :create-group-channels]
  [{:keys [interaction arguments]} auth metadata])

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Creating, leaving, and editing groups groups
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmethod slash/handle-slash-command-interaction [:groups :create]
  [{:keys [interaction arguments]} auth metadata]
  (if-let [guild-id (:guild-id interaction)]
    (let [formatted-name (str "grp-" (:name arguments))
          response (add-group-to-guild! auth guild-id formatted-name (:description arguments))]
      (i/channel-message-response interaction auth "Group created!" nil nil))
    (i/channel-message-response interaction auth "Please run this command in a guild!" nil nil)))

(defmethod slash/handle-slash-command-interaction [:groups :join]
  [{:keys [interaction arguments]} auth metadata]
  (let [guild-id (:guild-id interaction)
        available-groups (get @groups-by-guild guild-id)
        components (->> (vals available-groups)
                        (map (fn [{:keys [group-name role-id description]}]
                               (comps/menu-option group-name role-id :description description)))
                        (comps/select-menu :join-group)
                        (comps/action-row))]
    (clojure.pprint/pprint components)
    (i/channel-message-response interaction auth "**Available groups**" [components] nil)))

(defmethod slash/handle-slash-command-interaction [:groups :modify]
  [{:keys [interaction arguments]} auth metadata])

(defmethod slash/handle-slash-command-interaction [:groups :leave]
  [{:keys [interaction arguments]} auth metadata])

(defmethod slash/handle-slash-command-interaction [:groups :list]
  [{:keys [interaction]} auth metadata]
  (if-let [guild-id (:guild-id interaction)]
    (let [groups (get @groups-by-guild guild-id)
          message (if (seq groups)
                    (->> (keys groups)
                         (map (fn [g] (format "`%s`" g)))
                         (s/join ", ")
                         (format "**Available groups:** %s"))
                    "No available groups. Create one with `/groups create`!")]
      (i/channel-message-response interaction auth message nil nil))
    (i/channel-message-response interaction auth "Please run this command in a guild!" nil nil)))
