(ns aws-simple-sign.file-settings
  "Copied from cognitect.aws.config"
  (:require [clojure.string :as str]))

;;; predicates

(defn comment? [s]
  (str/starts-with? s "#"))

(defn start-profile? [s]
  (str/starts-with? s "["))

(defn start-nested? [s]
  (re-find #"^[\w-_\.]+\s*=$" s))

(defn add-profile-kv? [s]
  (re-find #"^[\w-_\.]+\s*=.*\w+" s))

(defn add-nested-kv? [s]
  (re-find #"^\s+[\w-_\.]+\s*=.*\w+" s))

;;; helpers

(defn split-lines [s]
  (into []
        (comp (map str/trimr)
              (remove str/blank?)
              (remove comment?))
        (str/split-lines s)))

(defn split-kv [s]
  (->> (str/split s #"=" 2)
       (map str/trim)))

;;; actions

(defn set-profile-path [m line]
  (assoc m :path [:profiles (second (re-find #"\[(?:profile)?\s*(.+)\]" line))]))

(defn ensure-profile-path [m]
  (update m :path (comp vec (partial take 2))))

(defn set-nested-path [m line]
  (update m :path #(conj (vec (take 2 %))
                         (-> line (str/replace #"=" "") (str/trim)))))

(defn add-profile-kv [m line]
  (let [[k v] (split-kv line)]
    (update-in m (take 2 (:path m)) assoc k v)))

(defn add-nested-kv [m line]
  (let [[k v] (split-kv line)]
    (update-in m (:path m) assoc k v)))

;;; main

(defn parse-content
  "Return the profiles in the configuration file."
  [content]
  (->> content
       split-lines
       (reduce (fn [m ^String line]
                 (cond (start-profile? line)
                       (set-profile-path m line)

                       (start-nested? line)
                       (set-nested-path m line)

                       (add-profile-kv? line)
                       (-> m
                           (add-profile-kv line)
                           (ensure-profile-path))

                       (add-nested-kv? line)
                       (add-nested-kv m line)

                       :else
                       (throw (ex-info "Invalid format in config" {}))))
               {:profiles {}})
       :profiles))

(comment
  (def c "[default]
region = us-east-1

[tardigrade]
region = us-west-1

[profile waterbear]
region = eu-west-1
credential_process=awsprocesscreds-saml -e 'https://url.com?with-query-string=true' -u some-user -a arn:aws:iam::1234:role/system/specialness

[nested]
s3 =
  max_concurrent_requests=10
  s3_key = s3_val
region = eu-west-1
s3 =
  max_queue_size=1000
foo.bar = baz  

[temp-credentials]
aws_session_token = FQoG/Ehj40mh/xf0TR+xLl+cp/xGWC+haIy+fJh6/fD+LFW=")
  
  (parse-content c)
  )