(ns dn.service-detect.core)

(defn- key-like
  "Finds a key in the map that matches given regex"
  [re m]
  (->> m
       (filter (comp (partial re-matches re) name first))
       (first)))

(defn detect-env-settings [env]
  (some->> env
           :test-activemq-url
           (hash-map :type :env :url)))

(defn detect-gitlab-service [env]
  (when-let [[k v] (key-like #".*-port-61616-tcp$" env)]
    {:url v
     :type :gitlab}))

(defn find-env-var [env pattern]
  (->> env
       (keys)
       (filter (comp (partial re-matches pattern) name))
       (first)))

(defn- port->host [k]
  (some->> (re-matches #"^(.*)-service-port-.*$" (name k))
           (second)
           (format "%s-service-host")
           (keyword)))

(defn kubernetes? [env]
  (some? (key-like #".*service-host$" env)))

(defn detect-k8s-service
  "Given the env vars and a port name, attempts to extract the host and port number
   of a Kubernetes service.  It looks for an env var that looks like `.*-service-port-<port-name>`
   and extracts the service name from that.  Returns a map that contains the host and the port,
   or nil if no service could be found."
  [env port-name]
  (when-let [match (some->> port-name
                            (name)
                            (format "^.*%s$")
                            (re-pattern)
                            (find-env-var env))]
    {:host (get env (port->host match))
     :port (get env match)}))

(defn- extract-k8s-service-names [env]
  (->> env
       (keys)
       (map (comp (partial re-matches #"^(.*)-service-host$") name))
       (filter some?)
       (map second)))

(defn- k8s-service-keys [n]
  (->> ["port" "service-host" "service-port"]
       (map (partial str n "-"))
       (map keyword)
       (zipmap [:tcp :host :port])))

(defn- env->k8s-service
  "Extracts all useful information from kubernetes env using the given service name"
  [env n]
  [(keyword n) (->> (k8s-service-keys n)
                    (reduce-kv (fn [r k v]
                                 (assoc r k (get env v)))
                               {:environment :kubernetes
                                :name n}))])

(defn detect-k8s-services
  "Tries to detect all exposed kubernetes services"
  [env]
  (->> (extract-k8s-service-names env)
       (map (partial env->k8s-service env))
       (into {})))

(defn env->services
  "Takes the given input environment map (assuming each env var is converted into
   a kebab-cased keyword) and tries to extract exposed service information from
   them."
  [env]
  (if (kubernetes? env)
    (detect-k8s-services env)
    {}))
