(ns dn.service-detect.kubernetes
  "Kubernetes specific code"
  (:require [dn.service-detect.utils :as u]))

(defn- kubernetes? [env]
  (some? (u/key-like #".*service-host$" env)))

(defn- extract-k8s-service-names [env]
  (->> env
       (keys)
       (map (comp (partial re-matches #"^(.*)-service-host$") name))
       (filter some?)
       (map second)))

(defn- service-port-names
  "Extracts the various service port names and their values from the
   environment for given service, as a map."
  [env n]
  (let [pattern (re-pattern (str "^" n "-service-port-(.+)$"))]
    (->> (keys env)
         (map (comp (partial re-matches pattern) name))
         (filter some?)
         (map (fn [[k v]]
                [v (get env (keyword k))]))
         (into {}))))

(defn- get-port-info [env s n val]
  {:environment :kubernetes
   :port-name n
   :service-name s
   :port (u/try-parse val)
   :host (get env (keyword (str s "-port-" val "-tcp-addr")))
   :tcp (get env (keyword (str s "-port-" val "-tcp")))})

(defn- env->k8s-service
  "Extracts all useful information from kubernetes env using the given service name"
  [env n]
  (->> (service-port-names env n)
       (map #(get-port-info env n (first %) (second %)))))

(defn detect-k8s-services
  "Tries to detect all exposed kubernetes services"
  [env]
  (when (kubernetes? env)
    (->> (extract-k8s-service-names env)
         (mapcat (partial env->k8s-service env)))))
