(ns dn.test-helpers.hiccup
  "Checkers and other helper functions for hiccup-style structures"
  (:require [midje.sweet :refer :all]
            [clojure.walk :as w]))

(defn extract-element
  "Given a keyword, extracts the html element from it by separating it from the optional classes and id"
  [x]
  (some-> x
          (name)
          (clojure.string/split #"\.|#")
          (first)
          (keyword)))

(defchecker html-element?
  "Verifies that the given vector is the specified html element"
  [el]
  (checker [v] (= el (extract-element (first v)))))

(defn attributes
  "Extracts any html attributes from the tag"
  [tag]
  (let [s (second tag)]
    (when (map? s) s)))

(defn- id-from-key [k]
  (some-> k
          (name)
          (clojure.string/split #"#")
          (second)
          (clojure.string/split #"\.")
          (first)
          (keyword)))

(defn tag-id
  "Given a html tag, retrieves its id"
  [tag]
  (or (-> tag (attributes) :id)
      (id-from-key (first tag))))

(defn html-tag? [x]
  (and (vector? x)
       (keyword? (first x))))

(defn subtags
  "Returns a list of subtags of given tag"
  [x]  
  (->> (rest x)
       ;; Skip attributes map
       (drop-while map?)))

(defn find-tag
  "Finds the first match in the tree for given matcher function.  If the matcher is a keyword,
   then it will look for the html tag with given key."
  [matcher tree]
  ;; TODO Change into tail recursion for large structures (using `loop/recur` or `trampoline`)
  (when (html-tag? tree)
    (let [m (if (keyword? matcher) (html-element? matcher) matcher)]
      (if (m tree)
        tree
        (->> (subtags tree)
             (map (partial find-tag matcher))
             (filter some?)
             (first))))))

(defn find-tag-by-id
  "Tries to find the element with given id"
  [id tree]
  (find-tag (comp (partial = id) tag-id) tree))
