(ns dn.test-helpers.hiccup
  "Checkers and other helper functions for hiccup-style structures"
  (:require [midje.sweet :refer :all]
            [clojure.walk :as w]))

(defn extract-element
  "Given a keyword, extracts the html element from it by separating it from the optional classes and id"
  [x]
  (some-> x
          (name)
          (clojure.string/split #"\.|#")
          (first)
          (keyword)))

(defchecker html-element?
  "Checker that verifies that the given vector is the specified html element"
  [el]
  (checker [v] (= el (extract-element (first v)))))

(defn- as-matcher
  "If x is a keyword or a non-fn, returns equality function for x, otherwise return x"
  [x]
  (if (or (keyword? x) (not (fn? x)))
    (comp (partial = x) first)
    x))

(defn attributes
  "Extracts any html attributes from the tag"
  [tag]
  (let [s (second tag)]
    (when (map? s) s)))

(defn has-attribute?
  "Checks if `tag` contains attribute matched by `m`.  `m` can be a value, in which
   case the attribute name is matched against it, or a function that should accept
   the vector of `[attr-key attr-val]`.  Returns the first match, or `nil`."
  [tag m]
  (->> (attributes tag)
       (filter (as-matcher m))
       (first)))

(defn- id-from-key [k]
  (some-> k
          (name)
          (clojure.string/split #"#")
          (second)
          (clojure.string/split #"\.")
          (first)
          (keyword)))

(defn tag-id
  "Given a html tag, retrieves its id"
  [tag]
  (or (-> tag (attributes) :id)
      (id-from-key (first tag))))

(defn html-tag? [x]
  (and (vector? x)
       (keyword? (first x))))

(defn subtags
  "Returns a list of subtags of given tag"
  [x]  
  (->> (rest x)
       ;; Skip attributes map
       (drop-while map?)))

(defn find-tag
  "Finds the first match in the tree for given matcher function.  If the matcher is a keyword,
   then it will look for the html tag with given key."
  [matcher tree]
  ;; TODO Change into tail recursion for large structures (using `loop/recur` or `trampoline`)
  (when (html-tag? tree)
    ;; TODO Use as-matcher here
    (let [m (if (keyword? matcher) (html-element? matcher) matcher)]
      (if (m tree)
        tree
        (->> (subtags tree)
             (map (partial find-tag matcher))
             (filter some?)
             (first))))))

(defn find-tag-by-id
  "Tries to find the element with given id"
  [id tree]
  (find-tag (comp (partial = id) tag-id) tree))

(defn classes
  "Returns the classes given tag belongs to"
  [tag]
  (some-> (attributes tag)
          :class
          (clojure.string/split #" ")))

(defn has-class?
  "Checks if given tag belongs to the given class."
  [tag cl]
  (contains? (set (classes tag)) cl))
